const sqlite3 = require('sqlite3').verbose();
const path = require('path');
const bcrypt = require('bcrypt');

const dbPath = path.join(__dirname, '..', 'database.db');

// データベース接続
const db = new sqlite3.Database(dbPath);

// テーブル作成とデータ初期化
function initializeDatabase() {
    return new Promise((resolve, reject) => {
        db.serialize(async () => {
            try {
                console.log('データベース初期化を開始...');

                // テーブル作成
                await createTables();

                // 既存データ確認
                const existingDataCheck = await checkExistingData();

                if (existingDataCheck.hasData) {
                    console.log('既存データが見つかりました。ダミーデータの生成をスキップします。');
                    console.log(`- お知らせ: ${existingDataCheck.newsCount}件`);
                    console.log(`- 行程表: ${existingDataCheck.schedulesCount}件`);
                    console.log(`- 参加者: ${existingDataCheck.participantsCount}件`);
                    console.log(`- 必要情報: ${existingDataCheck.informationCount}件`);

                    // 管理者アカウントのみ確認・作成
                    await ensureAdminExists();

                } else {
                    console.log('既存データなし。初期データを作成します...');

                    // 初期データ挿入（初回のみ）
                    await insertInitialData();

                    console.log('初期データの作成が完了しました。');
                }

                console.log('データベース初期化完了！');
                resolve();
            } catch (error) {
                console.error('データベース初期化エラー:', error);
                reject(error);
            }
        });
    });
}

// テーブル作成
function createTables() {
    return new Promise((resolve, reject) => {
        const tables = [
            // 管理者テーブル
            `CREATE TABLE IF NOT EXISTS admins (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT UNIQUE NOT NULL,
                password TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )`,
            
            // お知らせテーブル
            `CREATE TABLE IF NOT EXISTS news (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                content TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )`,
            
            // 行程表テーブル
            `CREATE TABLE IF NOT EXISTS schedules (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL,
                time TEXT NOT NULL,
                content TEXT NOT NULL,
                location TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )`,
            
            // 参加者基本情報テーブル
            `CREATE TABLE IF NOT EXISTS participants (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                school TEXT NOT NULL,
                grade TEXT NOT NULL,
                team TEXT NOT NULL,
                photo TEXT,
                password TEXT,
                type TEXT DEFAULT 'student',
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )`,
            
            // 学生プロフィールテーブル
            `CREATE TABLE IF NOT EXISTS student_profiles (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                participant_id INTEGER UNIQUE,
                participation_comment TEXT,
                hobby_selfpr TEXT,
                prestudy_task TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (participant_id) REFERENCES participants (id)
            )`,
            
            // 必要情報テーブル
            `CREATE TABLE IF NOT EXISTS information (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                content TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )`
        ];

        let completed = 0;
        tables.forEach((sql, index) => {
            db.run(sql, (err) => {
                if (err) {
                    reject(err);
                    return;
                }
                completed++;
                if (completed === tables.length) {
                    resolve();
                }
            });
        });
    });
}

// 既存データ確認関数
async function checkExistingData() {
    return new Promise((resolve, reject) => {
        const counts = {};

        db.get('SELECT COUNT(*) as count FROM news', (err, row) => {
            if (err) return reject(err);
            counts.newsCount = row.count;

            db.get('SELECT COUNT(*) as count FROM schedules', (err, row) => {
                if (err) return reject(err);
                counts.schedulesCount = row.count;

                db.get('SELECT COUNT(*) as count FROM participants', (err, row) => {
                    if (err) return reject(err);
                    counts.participantsCount = row.count;

                    db.get('SELECT COUNT(*) as count FROM information', (err, row) => {
                        if (err) return reject(err);
                        counts.informationCount = row.count;

                        // いずれかのテーブルにデータがあればtrue
                        counts.hasData = counts.newsCount > 0 ||
                                        counts.schedulesCount > 0 ||
                                        counts.participantsCount > 0 ||
                                        counts.informationCount > 0;

                        resolve(counts);
                    });
                });
            });
        });
    });
}

// 管理者アカウント確認・作成
async function ensureAdminExists() {
    return new Promise((resolve, reject) => {
        db.get('SELECT * FROM admins WHERE username = ?', ['admin'], async (err, row) => {
            if (err) return reject(err);

            if (!row) {
                console.log('管理者アカウントが存在しません。作成します...');
                await createAdminAccount();
            } else {
                console.log('管理者アカウントは既に存在します。');
            }
            resolve();
        });
    });
}

// 管理者アカウント作成
async function createAdminAccount() {
    return new Promise(async (resolve, reject) => {
        try {
            const hashedPassword = await bcrypt.hash('dev2590', 10);

            db.run(
                'INSERT OR IGNORE INTO admins (username, password) VALUES (?, ?)',
                ['admin', hashedPassword],
                (err) => {
                    if (err) reject(err);
                    else {
                        console.log('管理者アカウント作成: admin / dev2590');
                        resolve();
                    }
                }
            );
        } catch (error) {
            reject(error);
        }
    });
}

// 初期データ挿入（初回のみ）
async function insertInitialData() {
    return new Promise(async (resolve, reject) => {
        try {
            // 管理者アカウント作成
            await createAdminAccount();

            // サンプルお知らせ
            const newsData = [
                {
                    title: '研修旅行開催のお知らせ',
                    content: '2025年度 国際ロータリー第２５９０地区インターアクト研修旅行を下記の通り開催いたします。<br><br>【開催期間】2025年12月26日（金）～28日（日）<br>【参加者】各クラブのインターアクト生<br><br>詳細は後日お知らせいたします。'
                },
                {
                    title: '事前説明会について',
                    content: '研修旅行の事前説明会を2025年12月6日に開催予定です。<br>参加者の皆さんは必ずご参加ください。<br><br>【日時】2025年12月6日<br>【内容】研修旅行の詳細説明、事前課題について'
                },
                {
                    title: '事前課題の提出について',
                    content: '研修旅行に向けての事前課題を用意しております。<br>期日までに必ず提出してください。<br><br>課題内容については説明会でお伝えします。'
                }
            ];

            newsData.forEach(news => {
                db.run(`INSERT INTO news (title, content) VALUES (?, ?)`,
                    [news.title, news.content]);
            });

            // サンプル参加者データ
            const participantsData = [
                { name: '田中太郎', school: '◯◯高等学校', grade: '2年', team: 'A班', password: 'student001' },
                { name: '佐藤花子', school: '△△高等学校', grade: '1年', team: 'A班', password: 'student002' },
                { name: '鈴木一郎', school: '□□高等学校', grade: '2年', team: 'B班', password: 'student003' },
                { name: '高橋美咲', school: '◇◇高等学校', grade: '1年', team: 'B班', password: 'student004' },
                { name: '伊藤健太', school: '☆☆高等学校', grade: '2年', team: 'C班', password: 'student005' },
                { name: '山田優子', school: '※※高等学校', grade: '1年', team: 'C班', password: 'student006' },
                { name: '渡邊大輔', school: '●●高等学校', grade: '2年', team: 'D班', password: 'student007' },
                { name: '加藤愛', school: '■■高等学校', grade: '1年', team: 'D班', password: 'student008' },
                { name: '中村翔太', school: '▲▲高等学校', grade: '2年', team: 'E班', password: 'student009' },
                { name: '小林麻衣', school: '♦♦高等学校', grade: '1年', team: 'E班', password: 'student010' }
            ];

            for (const participant of participantsData) {
                const hashedStudentPassword = await bcrypt.hash(participant.password, 10);
                
                db.run(`INSERT INTO participants (name, school, grade, team, password, type)
                        VALUES (?, ?, ?, ?, ?, ?)`,
                    [participant.name, participant.school, participant.grade,
                     participant.team, hashedStudentPassword, 'student'], function(err) {
                    if (!err) {
                        // 学生プロフィールの初期化
                        db.run(`INSERT INTO student_profiles (participant_id, participation_comment, hobby_selfpr, prestudy_task)
                                VALUES (?, ?, ?, ?)`,
                            [this.lastID, '', '', '']);
                    }
                });
            }

            // サンプル行程表
            const scheduleData = [
                { date: '2025-12-26', time: '09:00', content: '集合・出発式', location: '○○駅前' },
                { date: '2025-12-26', time: '11:00', content: 'バス移動', location: '' },
                { date: '2025-12-26', time: '13:00', content: '昼食', location: '△△レストラン' },
                { date: '2025-12-26', time: '15:00', content: '文化施設見学', location: '□□博物館' },
                { date: '2025-12-26', time: '18:00', content: 'ホテルチェックイン', location: '◇◇ホテル' },
                { date: '2025-12-26', time: '19:00', content: '夕食・懇親会', location: 'ホテル内レストラン' },
                
                { date: '2025-12-27', time: '07:00', content: '起床・朝食', location: 'ホテル' },
                { date: '2025-12-27', time: '09:00', content: 'ワークショップ開始', location: '会議室A' },
                { date: '2025-12-27', time: '12:00', content: '昼食', location: 'ホテル内レストラン' },
                { date: '2025-12-27', time: '13:30', content: 'グループディスカッション', location: '会議室B' },
                { date: '2025-12-27', time: '16:00', content: '成果発表', location: '大会議室' },
                { date: '2025-12-27', time: '18:00', content: '夕食', location: 'ホテル内レストラン' },
                
                { date: '2025-12-28', time: '08:00', content: '朝食・チェックアウト', location: 'ホテル' },
                { date: '2025-12-28', time: '10:00', content: '地域交流活動', location: '☆☆コミュニティセンター' },
                { date: '2025-12-28', time: '12:00', content: '昼食', location: '※※食堂' },
                { date: '2025-12-28', time: '14:00', content: '帰路へ', location: '' },
                { date: '2025-12-28', time: '17:00', content: '到着・解散式', location: '○○駅前' }
            ];

            scheduleData.forEach(schedule => {
                db.run(`INSERT INTO schedules (date, time, content, location) VALUES (?, ?, ?, ?)`,
                    [schedule.date, schedule.time, schedule.content, schedule.location]);
            });

            // 必要情報
            const informationData = [
                {
                    title: '持参物',
                    content: '・着替え（2日分）<br>・洗面用具<br>・タオル<br>・筆記用具<br>・健康保険証のコピー<br>・その他個人で必要なもの'
                },
                {
                    title: '注意事項',
                    content: '・集合時間は厳守してください<br>・体調管理に十分注意してください<br>・貴重品の管理は各自で行ってください<br>・緊急時の連絡先を保護者に伝えておいてください'
                }
            ];

            informationData.forEach(info => {
                db.run(`INSERT INTO information (title, content) VALUES (?, ?)`,
                    [info.title, info.content]);
            });

            setTimeout(() => {
                resolve();
            }, 1000);
            
        } catch (error) {
            reject(error);
        }
    });
}

// データベース接続のヘルパー関数
function getDatabase() {
    return new sqlite3.Database(dbPath);
}

module.exports = {
    initializeDatabase,
    getDatabase
};

// スクリプトが直接実行された場合の処理
if (require.main === module) {
    initializeDatabase()
        .then(() => {
            console.log('データベース処理が完了しました');
            process.exit(0);
        })
        .catch((error) => {
            console.error('エラー:', error);
            process.exit(1);
        });
}