const express = require('express');
const bcrypt = require('bcrypt');
const { getDatabase } = require('../scripts/database');
const router = express.Router();

// 学生認証ミドルウェア（管理者または本人のみ許可）
function requireStudentAuth(req, res, next) {
    const studentId = parseInt(req.params.id);

    // 管理者の場合は全学生のプロフィール編集可能
    if (req.session.isAdmin) {
        return next();
    }

    // 学生本人の場合のみ編集可能
    if (req.session.studentId && req.session.studentId === studentId) {
        return next();
    }

    // 未認証の場合はメインページへリダイレクト
    return res.redirect('/?error=unauthorized');
}

// POST /student/login - ログイン処理（パスワードのみ）
router.post('/login', (req, res) => {
    const { studentId, password } = req.body;
    const db = getDatabase();

    if (!password) {
        return db.get(
            'SELECT * FROM participants WHERE id = ? AND type = "student"',
            [studentId],
            (err, student) => {
                db.close();
                res.render('student/login', {
                    student: student || {},
                    error: 'パスワードを入力してください'
                });
            }
        );
    }

    db.get(
        'SELECT * FROM participants WHERE id = ? AND type = "student"',
        [studentId],
        async (err, student) => {
            if (err || !student) {
                db.close();
                return res.redirect('/#student-login');
            }

            try {
                const match = await bcrypt.compare(password, student.password);
                if (match) {
                    req.session.studentId = student.id;
                    db.close();
                    return res.redirect(`/student/${student.id}/edit`);
                } else {
                    db.close();
                    return res.render('student/login', {
                        student,
                        error: 'パスワードが間違っています'
                    });
                }
            } catch (error) {
                db.close();
                return res.render('student/login', {
                    student,
                    error: 'ログインに失敗しました'
                });
            }
        }
    );
});

// グローバル学生ログアウト
router.get('/logout', (req, res) => {
    req.session.studentId = null;
    req.session.studentName = null;
    res.redirect('/');
});

// GET /student/:id/login - 学生個別のログインページ
router.get('/:id/login', (req, res) => {
    const studentId = parseInt(req.params.id);
    const db = getDatabase();

    db.get(
        'SELECT * FROM participants WHERE id = ? AND type = "student"',
        [studentId],
        (err, student) => {
            if (err || !student) {
                db.close();
                return res.redirect('/#student-login');
            }
            res.render('student/login', { student, error: null });
            db.close();
        }
    );
});

// 学生ログイン処理
router.post('/:id/login', async (req, res) => {
    try {
        const { id } = req.params;
        const { password, redirect } = req.body;
        const redirectUrl = redirect || `/student/${id}`;

        if (!/^\d+$/.test(id)) {
            return res.status(400).render('error', {
                title: 'エラー',
                message: '無効な学生IDです。',
                error: { status: 400 }
            });
        }

        const db = getDatabase();

        db.get('SELECT * FROM participants WHERE id = ? AND type = "student"', [id], async (err, student) => {
            if (err) {
                console.error('学生ログインエラー:', err);
                return res.render('student/login', {
                    title: 'ログイン',
                    studentId: id,
                    studentName: '学生',
                    redirect: redirectUrl,
                    error: 'システムエラーが発生しました。'
                });
            }

            if (!student || !student.password || !(await bcrypt.compare(password, student.password))) {
                return res.render('student/login', {
                    title: `${student ? student.name : '学生'}さんのログイン`,
                    studentId: id,
                    studentName: student ? student.name : '学生',
                    redirect: redirectUrl,
                    error: 'パスワードが正しくありません。'
                });
            }

            req.session.studentId = student.id;
            res.redirect(redirectUrl);
            db.close();
        });
    } catch (error) {
        console.error('学生ログイン処理エラー:', error);
        res.render('student/login', {
            title: 'ログイン',
            studentId: req.params.id,
            studentName: '学生',
            redirect: `/student/${req.params.id}`,
            error: 'システムエラーが発生しました。'
        });
    }
});

// 学生ログアウト
router.get('/:id/logout', (req, res) => {
    req.session.studentId = null;
    req.session.studentName = null;
    res.redirect(`/student/${req.params.id}`);
});

// 学生プロフィール閲覧ページ（閲覧専用）
router.get('/:id', (req, res) => {
    const studentId = parseInt(req.params.id);
    const db = getDatabase();

    db.get(`
        SELECT p.*, s.participation_comment, s.hobby_selfpr, s.prestudy_task
        FROM participants p
        LEFT JOIN student_profiles s ON p.id = s.participant_id
        WHERE p.id = ? AND p.type = 'student'
    `, [studentId], (err, student) => {
        if (err) {
            return res.status(500).render('error', {
                title: 'エラー',
                message: 'データベースエラーが発生しました',
                error: err
            });
        }

        if (!student) {
            return res.status(404).render('error', {
                title: '学生が見つかりません',
                message: '学生が見つかりません',
                error: { status: 404 }
            });
        }

        // 閲覧専用ページなので、編集関連の変数は渡さない
        res.render('student-profile', {
            student
        });
        db.close();
    });
});

// 学生プロフィール編集ページ（認証必須）
router.get('/:id/edit', requireStudentAuth, async (req, res) => {
    try {
        const { id } = req.params;
        const db = getDatabase();
        
        db.get(`
            SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
            FROM participants p 
            LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
            WHERE p.id = ? AND p.type = "student"
        `, [id], (err, student) => {
            if (err) {
                console.error('学生プロフィール取得エラー:', err);
                return res.status(500).render('error', {
                    title: 'エラー',
                    message: 'プロフィールの取得に失敗しました。',
                    error: err
                });
            }
            
            if (!student) {
                return res.status(404).render('error', {
                    title: '学生が見つかりません',
                    message: '指定された学生は存在しません。',
                    error: { status: 404 }
                });
            }
            
            res.render('student-edit', {
                title: `${student.name}さんのプロフィール編集`,
                student,
                isAdmin: req.session && req.session.isAdmin,
                error: null
            });
            db.close();
        });
    } catch (error) {
        console.error('学生プロフィール編集ページエラー:', error);
        res.status(500).render('error', {
            title: 'エラー',
            message: 'システムエラーが発生しました。',
            error: error
        });
    }
});

// 学生プロフィール更新処理（認証必須）
router.post('/:id/update', requireStudentAuth, async (req, res) => {
    try {
        const { id } = req.params;
        const { participation_comment, hobby_selfpr, prestudy_task } = req.body;
        const db = getDatabase();
        
        // まず学生プロフィールが存在するかチェック
        db.get('SELECT * FROM student_profiles WHERE participant_id = ?', [id], (err, profile) => {
            if (err) {
                console.error('プロフィールチェックエラー:', err);
                return res.status(500).json({ error: 'システムエラーが発生しました。' });
            }
            
            if (profile) {
                // 更新
                db.run('UPDATE student_profiles SET participation_comment = ?, hobby_selfpr = ?, prestudy_task = ?, updated_at = CURRENT_TIMESTAMP WHERE participant_id = ?', 
                       [participation_comment, hobby_selfpr, prestudy_task, id], function(err) {
                    if (err) {
                        console.error('プロフィール更新エラー:', err);
                        return res.status(500).json({ error: 'プロフィールの更新に失敗しました。' });
                    }
                    
                    res.redirect(`/student/${id}?updated=true`);
                    db.close();
                });
            } else {
                // 新規作成
                db.run('INSERT INTO student_profiles (participant_id, participation_comment, hobby_selfpr, prestudy_task) VALUES (?, ?, ?, ?)', 
                       [id, participation_comment, hobby_selfpr, prestudy_task], function(err) {
                    if (err) {
                        console.error('プロフィール作成エラー:', err);
                        return res.status(500).json({ error: 'プロフィールの作成に失敗しました。' });
                    }
                    
                    res.redirect(`/student/${id}?updated=true`);
                    db.close();
                });
            }
        });
    } catch (error) {
        console.error('プロフィール更新処理エラー:', error);
        
        // エラー時は編集ページに戻る
        try {
            const db = getDatabase();
            const student = await new Promise((resolve, reject) => {
                db.get(`
                    SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
                    FROM participants p 
                    LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
                    WHERE p.id = ? AND p.type = "student"
                `, [id], (err, result) => {
                    db.close();
                    if (err) reject(err);
                    else resolve(result);
                });
            });
            
            res.render('student-edit', { 
                title: `${student.name}さんのプロフィール編集`,
                student,
                error: 'プロフィールの更新に失敗しました。しばらくしてからお試しください。'
            });
        } catch (getError) {
            console.error('学生データ取得エラー:', getError);
            res.status(500).render('error', {
                title: 'エラー',
                message: 'システムエラーが発生しました。',
                error: { status: 500 }
            });
        }
    }
});

module.exports = router;