const express = require('express');
const bcrypt = require('bcrypt');
const multer = require('multer');
const sharp = require('sharp');
const path = require('path');
const fs = require('fs').promises;
const { getDatabase } = require('../scripts/database');
const router = express.Router();

// 画像アップロード設定
const storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, path.join(__dirname, '..', 'public', 'uploads'));
    },
    filename: function (req, file, cb) {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, 'participant-' + uniqueSuffix + path.extname(file.originalname));
    }
});
const upload = multer({ 
    storage: storage,
    limits: { fileSize: 5 * 1024 * 1024 }, // 5MB制限
    fileFilter: (req, file, cb) => {
        const allowedTypes = /jpeg|jpg|png|gif/;
        const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
        const mimetype = allowedTypes.test(file.mimetype);
        
        if (mimetype && extname) {
            return cb(null, true);
        } else {
            cb(new Error('画像ファイル（JPEG、PNG、GIF）のみアップロード可能です。'));
        }
    }
});

// 管理者認証ミドルウェア
function requireAuth(req, res, next) {
    if (!req.session.adminId) {
        return res.redirect('/admin/login');
    }
    next();
}

// ログインページ
router.get('/login', (req, res) => {
    if (req.session.adminId) {
        return res.redirect('/admin');
    }
    res.render('admin/login', { title: '管理者ログイン', error: null });
});

// ログイン処理
router.post('/login', async (req, res) => {
    try {
        const { username, password } = req.body;
        const db = getDatabase();
        
        db.get('SELECT * FROM admins WHERE username = ?', [username], async (err, admin) => {
            if (err) {
                console.error('ログインエラー:', err);
                return res.render('admin/login', { 
                    title: '管理者ログイン', 
                    error: 'システムエラーが発生しました。' 
                });
            }
            
            if (!admin || !(await bcrypt.compare(password, admin.password))) {
                return res.render('admin/login', { 
                    title: '管理者ログイン', 
                    error: 'ユーザー名またはパスワードが正しくありません。' 
                });
            }
            
            req.session.adminId = admin.id;
            req.session.isAdmin = true;
            res.redirect('/admin');
            db.close();
        });
    } catch (error) {
        console.error('ログインエラー:', error);
        res.render('admin/login', { 
            title: '管理者ログイン', 
            error: 'システムエラーが発生しました。' 
        });
    }
});

// ログアウト
router.post('/logout', (req, res) => {
    req.session.destroy();
    res.redirect('/admin/login');
});

// ダッシュボード
router.get('/', requireAuth, async (req, res) => {
    try {
        const db = getDatabase();
        
        const [newsCount, participantsCount, schedulesCount] = await Promise.all([
            new Promise((resolve, reject) => {
                db.get('SELECT COUNT(*) as count FROM news', (err, row) => {
                    if (err) reject(err);
                    else resolve(row.count);
                });
            }),
            new Promise((resolve, reject) => {
                db.get('SELECT COUNT(*) as count FROM participants', (err, row) => {
                    if (err) reject(err);
                    else resolve(row.count);
                });
            }),
            new Promise((resolve, reject) => {
                db.get('SELECT COUNT(*) as count FROM schedules', (err, row) => {
                    if (err) reject(err);
                    else resolve(row.count);
                });
            })
        ]);
        
        db.close();
        
        res.render('admin/dashboard', {
            title: '管理画面ダッシュボード',
            stats: {
                news: newsCount,
                participants: participantsCount,
                schedules: schedulesCount
            }
        });
    } catch (error) {
        console.error('ダッシュボードエラー:', error);
        res.status(500).render('error', {
            title: 'エラー',
            message: 'データの取得に失敗しました。',
            error: error
        });
    }
});

// お知らせ管理
router.get('/news', requireAuth, async (req, res) => {
    try {
        const db = getDatabase();
        
        db.all('SELECT * FROM news ORDER BY created_at DESC', (err, news) => {
            if (err) {
                console.error('お知らせ取得エラー:', err);
                return res.status(500).render('error', {
                    title: 'エラー',
                    message: 'お知らせの取得に失敗しました。',
                    error: err
                });
            }
            
            res.render('admin/news', { title: 'お知らせ管理', news });
            db.close();
        });
    } catch (error) {
        console.error('お知らせ管理エラー:', error);
        res.status(500).render('error', {
            title: 'エラー',
            message: 'システムエラーが発生しました。',
            error: error
        });
    }
});

// お知らせ追加処理
router.post('/news', requireAuth, async (req, res) => {
    try {
        const { title, content } = req.body;
        const db = getDatabase();
        
        db.run('INSERT INTO news (title, content) VALUES (?, ?)', [title, content], function(err) {
            if (err) {
                console.error('お知らせ追加エラー:', err);
                return res.status(500).json({ error: 'お知らせの追加に失敗しました。' });
            }
            
            res.json({ success: true, id: this.lastID });
            db.close();
        });
    } catch (error) {
        console.error('お知らせ追加エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// お知らせ更新処理
router.put('/news/:id', requireAuth, async (req, res) => {
    try {
        const { id } = req.params;
        const { title, content } = req.body;
        const db = getDatabase();
        
        db.run('UPDATE news SET title = ?, content = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?', 
               [title, content, id], function(err) {
            if (err) {
                console.error('お知らせ更新エラー:', err);
                return res.status(500).json({ error: 'お知らせの更新に失敗しました。' });
            }
            
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('お知らせ更新エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// お知らせ削除処理
router.delete('/news/:id', requireAuth, async (req, res) => {
    try {
        const { id } = req.params;
        console.log('🗑️ お知らせ削除リクエスト - ID:', id);
        const db = getDatabase();
        
        db.run('DELETE FROM news WHERE id = ?', [id], function(err) {
            if (err) {
                console.error('お知らせ削除エラー:', err);
                return res.status(500).json({ error: 'お知らせの削除に失敗しました。' });
            }
            
            console.log('✅ お知らせ削除成功 - ID:', id, '削除件数:', this.changes);
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('お知らせ削除エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 参加者管理
router.get('/participants', requireAuth, async (req, res) => {
    try {
        const db = getDatabase();
        
        db.all('SELECT * FROM participants ORDER BY team, name', (err, participants) => {
            if (err) {
                console.error('参加者取得エラー:', err);
                return res.status(500).render('error', {
                    title: 'エラー',
                    message: '参加者の取得に失敗しました。',
                    error: err
                });
            }
            
            res.render('admin/participants', { title: '参加者管理', participants });
            db.close();
        });
    } catch (error) {
        console.error('参加者管理エラー:', error);
        res.status(500).render('error', {
            title: 'エラー',
            message: 'システムエラーが発生しました。',
            error: error
        });
    }
});

// 参加者追加処理
router.post('/participants', requireAuth, upload.single('photo'), async (req, res) => {
    try {
        const { name, school, grade, team, password, type } = req.body;
        let photoPath = null;
        
        // 画像処理
        if (req.file) {
            const resizedImagePath = path.join(__dirname, '..', 'public', 'uploads', 'resized-' + req.file.filename);
            await sharp(req.file.path)
                .resize(300, 300, { fit: 'cover' })
                .jpeg({ quality: 85 })
                .toFile(resizedImagePath);
            
            // 元の画像を削除し、リサイズ後の画像をリネーム
            await fs.unlink(req.file.path);
            await fs.rename(resizedImagePath, req.file.path);
            photoPath = '/uploads/' + req.file.filename;
        }
        
        const db = getDatabase();
        const hashedPassword = password ? await bcrypt.hash(password, 10) : null;
        
        db.run('INSERT INTO participants (name, school, grade, team, photo, password, type) VALUES (?, ?, ?, ?, ?, ?, ?)', 
               [name, school, grade, team, photoPath, hashedPassword, type], function(err) {
            if (err) {
                console.error('参加者追加エラー:', err);
                return res.status(500).json({ error: '参加者の追加に失敗しました。' });
            }
            
            // 学生の場合、プロフィールテーブルにも初期データを追加
            if (type === 'student') {
                db.run('INSERT INTO student_profiles (participant_id, participation_comment, hobby_selfpr, prestudy_task) VALUES (?, ?, ?, ?)', 
                       [this.lastID, '', '', '']);
            }
            
            res.json({ success: true, id: this.lastID });
            db.close();
        });
    } catch (error) {
        console.error('参加者追加エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 参加者更新処理
router.put('/participants/:id', requireAuth, upload.single('photo'), async (req, res) => {
    try {
        const { id } = req.params;
        const { name, school, grade, team, password, type } = req.body;
        const db = getDatabase();
        
        let updateQuery = 'UPDATE participants SET name = ?, school = ?, grade = ?, team = ?, type = ?';
        let updateParams = [name, school, grade, team, type];
        
        // パスワードが提供された場合
        if (password) {
            const hashedPassword = await bcrypt.hash(password, 10);
            updateQuery += ', password = ?';
            updateParams.push(hashedPassword);
        }
        
        // 画像が提供された場合
        if (req.file) {
            const resizedImagePath = path.join(__dirname, '..', 'public', 'uploads', 'resized-' + req.file.filename);
            await sharp(req.file.path)
                .resize(300, 300, { fit: 'cover' })
                .jpeg({ quality: 85 })
                .toFile(resizedImagePath);
            
            await fs.unlink(req.file.path);
            await fs.rename(resizedImagePath, req.file.path);
            
            updateQuery += ', photo = ?';
            updateParams.push('/uploads/' + req.file.filename);
        }
        
        updateQuery += ', updated_at = CURRENT_TIMESTAMP WHERE id = ?';
        updateParams.push(id);
        
        db.run(updateQuery, updateParams, function(err) {
            if (err) {
                console.error('参加者更新エラー:', err);
                return res.status(500).json({ error: '参加者の更新に失敗しました。' });
            }
            
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('参加者更新エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 参加者削除処理
router.delete('/participants/:id', requireAuth, async (req, res) => {
    try {
        const { id } = req.params;
        console.log('🗑️ 参加者削除リクエスト - ID:', id);
        const db = getDatabase();
        
        // 関連する学生プロフィールも削除
        db.run('DELETE FROM student_profiles WHERE participant_id = ?', [id]);
        db.run('DELETE FROM participants WHERE id = ?', [id], function(err) {
            if (err) {
                console.error('参加者削除エラー:', err);
                return res.status(500).json({ error: '参加者の削除に失敗しました。' });
            }
            
            console.log('✅ 参加者削除成功 - ID:', id, '削除件数:', this.changes);
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('参加者削除エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 行程表管理
router.get('/schedules', requireAuth, async (req, res) => {
    try {
        const db = getDatabase();
        
        db.all('SELECT * FROM schedules ORDER BY date ASC, time ASC', (err, schedules) => {
            if (err) {
                console.error('行程表取得エラー:', err);
                return res.status(500).render('error', {
                    title: 'エラー',
                    message: '行程表の取得に失敗しました。',
                    error: err
                });
            }
            
            res.render('admin/schedules', { title: '行程表管理', schedules });
            db.close();
        });
    } catch (error) {
        console.error('行程表管理エラー:', error);
        res.status(500).render('error', {
            title: 'エラー',
            message: 'システムエラーが発生しました。',
            error: error
        });
    }
});

// 行程表追加処理
router.post('/schedules', requireAuth, async (req, res) => {
    try {
        const { date, time, content, location } = req.body;
        const db = getDatabase();
        
        db.run('INSERT INTO schedules (date, time, content, location) VALUES (?, ?, ?, ?)', 
               [date, time, content, location], function(err) {
            if (err) {
                console.error('行程表追加エラー:', err);
                return res.status(500).json({ error: '行程表の追加に失敗しました。' });
            }
            
            res.json({ success: true, id: this.lastID });
            db.close();
        });
    } catch (error) {
        console.error('行程表追加エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 行程表更新処理
router.put('/schedules/:id', requireAuth, async (req, res) => {
    try {
        const { id } = req.params;
        const { date, time, content, location } = req.body;
        const db = getDatabase();
        
        db.run('UPDATE schedules SET date = ?, time = ?, content = ?, location = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?', 
               [date, time, content, location, id], function(err) {
            if (err) {
                console.error('行程表更新エラー:', err);
                return res.status(500).json({ error: '行程表の更新に失敗しました。' });
            }
            
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('行程表更新エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 行程表削除処理
router.delete('/schedules/:id', requireAuth, async (req, res) => {
    try {
        const { id } = req.params;
        console.log('🗑️ 行程表削除リクエスト - ID:', id);
        const db = getDatabase();
        
        db.run('DELETE FROM schedules WHERE id = ?', [id], function(err) {
            if (err) {
                console.error('行程表削除エラー:', err);
                return res.status(500).json({ error: '行程表の削除に失敗しました。' });
            }
            
            console.log('✅ 行程表削除成功 - ID:', id, '削除件数:', this.changes);
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('行程表削除エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 必要情報管理
router.get('/information', requireAuth, async (req, res) => {
    try {
        const db = getDatabase();
        
        db.all('SELECT * FROM information ORDER BY created_at DESC', (err, information) => {
            if (err) {
                console.error('必要情報取得エラー:', err);
                return res.status(500).render('error', {
                    title: 'エラー',
                    message: '必要情報の取得に失敗しました。',
                    error: err
                });
            }
            
            res.render('admin/information', { title: '必要情報管理', information });
            db.close();
        });
    } catch (error) {
        console.error('必要情報管理エラー:', error);
        res.status(500).render('error', {
            title: 'エラー',
            message: 'システムエラーが発生しました。',
            error: error
        });
    }
});

// 必要情報追加処理
router.post('/information', requireAuth, async (req, res) => {
    try {
        const { title, content } = req.body;
        const db = getDatabase();
        
        db.run('INSERT INTO information (title, content) VALUES (?, ?)', [title, content], function(err) {
            if (err) {
                console.error('必要情報追加エラー:', err);
                return res.status(500).json({ error: '必要情報の追加に失敗しました。' });
            }
            
            res.json({ success: true, id: this.lastID });
            db.close();
        });
    } catch (error) {
        console.error('必要情報追加エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 必要情報更新処理
router.put('/information/:id', requireAuth, async (req, res) => {
    try {
        const { id } = req.params;
        const { title, content } = req.body;
        const db = getDatabase();
        
        db.run('UPDATE information SET title = ?, content = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?', 
               [title, content, id], function(err) {
            if (err) {
                console.error('必要情報更新エラー:', err);
                return res.status(500).json({ error: '必要情報の更新に失敗しました。' });
            }
            
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('必要情報更新エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 必要情報削除処理
router.delete('/information/:id', requireAuth, async (req, res) => {
    try {
        const { id } = req.params;
        console.log('🗑️ 必要情報削除リクエスト - ID:', id);
        const db = getDatabase();
        
        db.run('DELETE FROM information WHERE id = ?', [id], function(err) {
            if (err) {
                console.error('必要情報削除エラー:', err);
                return res.status(500).json({ error: '必要情報の削除に失敗しました。' });
            }
            
            console.log('✅ 必要情報削除成功 - ID:', id, '削除件数:', this.changes);
            res.json({ success: true });
            db.close();
        });
    } catch (error) {
        console.error('必要情報削除エラー:', error);
        res.status(500).json({ error: 'システムエラーが発生しました。' });
    }
});

// 一括削除API
router.delete('/api/bulk-delete', requireAuth, async (req, res) => {
    try {
        const { type, ids } = req.body;
        console.log('🗑️ 一括削除リクエスト - Type:', type, 'IDs:', ids);
        
        if (!ids || !Array.isArray(ids) || ids.length === 0) {
            return res.status(400).json({ 
                success: false, 
                message: '削除するIDが指定されていません' 
            });
        }
        
        // テーブル名の検証（SQLインジェクション対策）
        const allowedTypes = ['news', 'schedules', 'participants', 'information'];
        if (!allowedTypes.includes(type)) {
            return res.status(400).json({ 
                success: false, 
                message: '無効なテーブル名です' 
            });
        }
        
        // IDsを安全な形式に変換
        const safeIds = ids.map(id => parseInt(id)).filter(id => !isNaN(id));
        if (safeIds.length === 0) {
            return res.status(400).json({ 
                success: false, 
                message: '有効なIDがありません' 
            });
        }
        
        const db = getDatabase();
        
        // 参加者の場合は関連データも削除
        if (type === 'participants') {
            // 関連する学生プロフィールも削除
            const placeholders = safeIds.map(() => '?').join(',');
            db.run(`DELETE FROM student_profiles WHERE participant_id IN (${placeholders})`, safeIds);
        }
        
        // プレースホルダーを作成（?,?,?...）
        const placeholders = safeIds.map(() => '?').join(',');
        const query = `DELETE FROM ${type} WHERE id IN (${placeholders})`;
        
        // データベース実行
        db.run(query, safeIds, function(err) {
            if (err) {
                console.error('一括削除エラー:', err);
                return res.status(500).json({ 
                    success: false, 
                    error: '一括削除に失敗しました' 
                });
            }
            
            console.log('✅ 一括削除成功 - 削除件数:', this.changes);
            res.json({ 
                success: true, 
                deletedCount: this.changes,
                message: `${this.changes}件のアイテムを削除しました`
            });
            db.close();
        });
        
    } catch (error) {
        console.error('一括削除エラー:', error);
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

module.exports = router;