// メインサイト用JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // 画像の遅延読み込み
    initLazyLoading();
    
    // スムーススクロール
    initSmoothScroll();
    
    // 印刷用スタイル調整
    initPrintStyles();
});

// 画像の遅延読み込み
function initLazyLoading() {
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    if (img.dataset.src) {
                        img.src = img.dataset.src;
                        img.removeAttribute('data-src');
                        img.classList.remove('lazy');
                        observer.unobserve(img);
                    }
                }
            });
        });

        document.querySelectorAll('img[data-src]').forEach(img => {
            imageObserver.observe(img);
        });
    }
}

// スムーススクロール
function initSmoothScroll() {
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
}

// 印刷用スタイル調整
function initPrintStyles() {
    window.addEventListener('beforeprint', function() {
        // 印刷時に不要な要素を非表示
        const elementsToHide = [
            '.admin-link',
            '.login-section',
            '.header .event-date'
        ];
        
        elementsToHide.forEach(selector => {
            const elements = document.querySelectorAll(selector);
            elements.forEach(el => {
                el.style.display = 'none';
            });
        });
    });
    
    window.addEventListener('afterprint', function() {
        // 印刷後に元に戻す
        const elementsToShow = [
            '.admin-link',
            '.login-section',
            '.header .event-date'
        ];
        
        elementsToShow.forEach(selector => {
            const elements = document.querySelectorAll(selector);
            elements.forEach(el => {
                el.style.display = '';
            });
        });
    });
}

// ユーティリティ関数
const utils = {
    // 日付フォーマット
    formatDate: function(dateString) {
        const date = new Date(dateString);
        return date.toLocaleDateString('ja-JP', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            weekday: 'short'
        });
    },
    
    // 時間フォーマット
    formatTime: function(timeString) {
        return timeString.replace(/^(\d{1,2}):(\d{2})/, '$1:$2');
    },
    
    // HTMLサニタイズ（基本的な処理）
    sanitizeHTML: function(str) {
        const div = document.createElement('div');
        div.textContent = str;
        return div.innerHTML;
    },
    
    // ローディング表示
    showLoading: function(element) {
        if (element) {
            element.classList.add('loading');
            element.style.opacity = '0.5';
            element.style.pointerEvents = 'none';
        }
    },
    
    // ローディング非表示
    hideLoading: function(element) {
        if (element) {
            element.classList.remove('loading');
            element.style.opacity = '';
            element.style.pointerEvents = '';
        }
    }
};

// グローバルに公開
window.siteUtils = utils;