// 管理画面用JavaScript

document.addEventListener('DOMContentLoaded', function() {
    // 共通初期化
    initCommonFeatures();
    
    // ページ別初期化
    initPageSpecificFeatures();
    
    // 編集機能の初期化
    initEditFeatures();
    
    // 一括削除機能の初期化
    initBulkDeleteFeatures();
});

// 共通機能の初期化
function initCommonFeatures() {
    // 確認ダイアログ
    initConfirmDialogs();
    
    // 自動保存機能（フォーム）
    initAutoSave();
    
    // キーボードショートカット
    initKeyboardShortcuts();
    
    // ツールチップ
    initTooltips();
}

// ページ別機能の初期化
function initPageSpecificFeatures() {
    const path = window.location.pathname;
    
    if (path.includes('/admin/participants')) {
        initParticipantsPage();
    } else if (path.includes('/admin/schedules')) {
        initSchedulesPage();
    } else if (path.includes('/admin/news')) {
        initNewsPage();
    } else if (path.includes('/admin/information')) {
        initInformationPage();
    }
}

// 確認ダイアログ
function initConfirmDialogs() {
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('btn-danger') || 
            e.target.closest('.btn-danger')) {
            const btn = e.target.classList.contains('btn-danger') ? 
                       e.target : e.target.closest('.btn-danger');
            
            if (btn.textContent.includes('削除')) {
                if (!confirm('本当に削除しますか？この操作は取り消せません。')) {
                    e.preventDefault();
                    return false;
                }
            }
        }
    });
}

// 自動保存機能
function initAutoSave() {
    const forms = document.querySelectorAll('form[data-autosave]');
    forms.forEach(form => {
        const inputs = form.querySelectorAll('input, textarea, select');
        inputs.forEach(input => {
            input.addEventListener('input', debounce(function() {
                const formData = new FormData(form);
                const data = Object.fromEntries(formData.entries());
                
                // localStorage に保存
                const key = `autosave_${form.dataset.autosave}`;
                localStorage.setItem(key, JSON.stringify(data));
                
                // 保存インジケーター表示
                showSaveIndicator('自動保存されました');
            }, 1000));
        });
        
        // ページロード時に復元
        const key = `autosave_${form.dataset.autosave}`;
        const saved = localStorage.getItem(key);
        if (saved) {
            try {
                const data = JSON.parse(saved);
                Object.keys(data).forEach(name => {
                    const input = form.querySelector(`[name="${name}"]`);
                    if (input) {
                        input.value = data[name];
                    }
                });
            } catch (e) {
                console.warn('自動保存データの復元に失敗:', e);
            }
        }
    });
}

// キーボードショートカット
function initKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // Ctrl+S で保存
        if (e.ctrlKey && e.key === 's') {
            e.preventDefault();
            const submitBtn = document.querySelector('form button[type="submit"]');
            if (submitBtn) {
                submitBtn.click();
            }
        }
        
        // Ctrl+N で新規作成
        if (e.ctrlKey && e.key === 'n') {
            e.preventDefault();
            const addBtn = document.querySelector('[id*="add"][id*="Btn"], [id*="Add"][id*="Btn"]');
            if (addBtn) {
                addBtn.click();
            }
        }
        
        // Escape でモーダルを閉じる
        if (e.key === 'Escape') {
            const modal = document.querySelector('.modal[style*="block"]');
            if (modal) {
                const closeBtn = modal.querySelector('.modal-close');
                if (closeBtn) {
                    closeBtn.click();
                }
            }
        }
    });
}

// ツールチップ
function initTooltips() {
    const tooltipElements = document.querySelectorAll('[data-tooltip]');
    tooltipElements.forEach(el => {
        el.addEventListener('mouseenter', showTooltip);
        el.addEventListener('mouseleave', hideTooltip);
    });
}

// 参加者ページ専用機能
function initParticipantsPage() {
    // 参加者種別の変更でパスワードフィールドの表示切替
    const typeSelect = document.querySelector('#participantType');
    if (typeSelect) {
        typeSelect.addEventListener('change', togglePasswordField);
        togglePasswordField.call(typeSelect);
    }
    
    // 画像プレビュー
    const photoInput = document.querySelector('#participantPhoto');
    if (photoInput) {
        photoInput.addEventListener('change', previewImage);
    }
}

// 行程表ページ専用機能
function initSchedulesPage() {
    // 時刻入力のバリデーション
    const timeInputs = document.querySelectorAll('input[type="time"]');
    timeInputs.forEach(input => {
        input.addEventListener('change', validateTimeInput);
    });
}

// ニュースページ専用機能
function initNewsPage() {
    // HTMLプレビュー機能
    const contentTextarea = document.querySelector('#newsContent');
    if (contentTextarea) {
        contentTextarea.addEventListener('input', debounce(showHTMLPreview, 500));
    }
}

// 必要情報ページ専用機能
function initInformationPage() {
    // HTMLプレビュー機能
    const contentTextarea = document.querySelector('#infoContent');
    if (contentTextarea) {
        contentTextarea.addEventListener('input', debounce(showHTMLPreview, 500));
    }
}

// パスワードフィールドの表示切替
function togglePasswordField() {
    const passwordGroup = document.querySelector('#passwordGroup');
    const passwordInput = document.querySelector('#participantPassword');
    
    if (this.value === 'student') {
        passwordGroup.style.display = 'block';
        passwordInput.required = true;
    } else {
        passwordGroup.style.display = 'none';
        passwordInput.required = false;
        passwordInput.value = '';
    }
}

// 画像プレビュー
function previewImage(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            // プレビュー要素があれば表示
            let preview = document.querySelector('#imagePreview');
            if (!preview) {
                preview = document.createElement('div');
                preview.id = 'imagePreview';
                preview.innerHTML = '<img id="previewImg" style="max-width: 200px; max-height: 200px; border-radius: 8px; margin-top: 10px;">';
                e.target.parentNode.appendChild(preview);
            }
            
            const img = preview.querySelector('#previewImg');
            img.src = e.target.result;
            preview.style.display = 'block';
        };
        reader.readAsDataURL(file);
    }
}

// 時刻入力のバリデーション
function validateTimeInput(e) {
    const time = e.target.value;
    if (time) {
        // 24時間形式をチェック
        const [hours, minutes] = time.split(':').map(Number);
        if (hours < 0 || hours > 23 || minutes < 0 || minutes > 59) {
            alert('正しい時刻を入力してください（00:00-23:59）');
            e.target.focus();
        }
    }
}

// HTMLプレビュー
function showHTMLPreview() {
    const textarea = this;
    const content = textarea.value;
    
    // プレビュー要素を作成または取得
    let preview = document.querySelector('#htmlPreview');
    if (!preview) {
        preview = document.createElement('div');
        preview.id = 'htmlPreview';
        preview.style.cssText = `
            border: 1px solid #ddd;
            border-radius: 6px;
            padding: 1rem;
            margin-top: 10px;
            background: #f9f9f9;
            max-height: 200px;
            overflow-y: auto;
        `;
        
        const label = document.createElement('label');
        label.textContent = 'プレビュー:';
        label.style.cssText = 'display: block; margin-top: 10px; font-weight: 600; color: #666;';
        
        textarea.parentNode.appendChild(label);
        textarea.parentNode.appendChild(preview);
    }
    
    // HTMLコンテンツを表示
    preview.innerHTML = content || '<em>プレビューがここに表示されます</em>';
}

// 保存インジケーター
function showSaveIndicator(message) {
    // 既存のインジケーターを削除
    const existing = document.querySelector('#saveIndicator');
    if (existing) {
        existing.remove();
    }
    
    // 新しいインジケーターを作成
    const indicator = document.createElement('div');
    indicator.id = 'saveIndicator';
    indicator.textContent = message;
    indicator.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #28a745;
        color: white;
        padding: 10px 20px;
        border-radius: 6px;
        z-index: 10000;
        font-size: 14px;
        font-weight: 500;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        transform: translateX(100%);
        transition: transform 0.3s ease;
    `;
    
    document.body.appendChild(indicator);
    
    // アニメーション
    setTimeout(() => {
        indicator.style.transform = 'translateX(0)';
    }, 10);
    
    setTimeout(() => {
        indicator.style.transform = 'translateX(100%)';
        setTimeout(() => {
            indicator.remove();
        }, 300);
    }, 2000);
}

// ツールチップ表示
function showTooltip(e) {
    const tooltip = document.createElement('div');
    tooltip.className = 'tooltip';
    tooltip.textContent = e.target.dataset.tooltip;
    tooltip.style.cssText = `
        position: absolute;
        background: #333;
        color: white;
        padding: 5px 10px;
        border-radius: 4px;
        font-size: 12px;
        white-space: nowrap;
        z-index: 10000;
        pointer-events: none;
        opacity: 0;
        transition: opacity 0.3s;
    `;
    
    document.body.appendChild(tooltip);
    
    const rect = e.target.getBoundingClientRect();
    tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';
    tooltip.style.top = (rect.top - tooltip.offsetHeight - 5) + 'px';
    
    setTimeout(() => {
        tooltip.style.opacity = '1';
    }, 10);
    
    e.target._tooltip = tooltip;
}

// ツールチップ非表示
function hideTooltip(e) {
    if (e.target._tooltip) {
        e.target._tooltip.remove();
        delete e.target._tooltip;
    }
}

// デバウンス関数
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func.apply(this, args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// ユーティリティ関数
const adminUtils = {
    // API呼び出し
    api: async function(url, options = {}) {
        const defaultOptions = {
            headers: {
                'Content-Type': 'application/json',
            },
        };
        
        const response = await fetch(url, { ...defaultOptions, ...options });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return response.json();
    },
    
    // フォームデータをJSONに変換
    formToJSON: function(form) {
        const formData = new FormData(form);
        const data = {};
        for (let [key, value] of formData.entries()) {
            data[key] = value;
        }
        return data;
    },
    
    // アラート表示
    showAlert: function(type, message, duration = 5000) {
        const alert = document.createElement('div');
        alert.className = `alert alert-${type}`;
        alert.textContent = message;
        alert.style.cssText = `
            position: fixed;
            top: 20px;
            left: 50%;
            transform: translateX(-50%);
            z-index: 10000;
            padding: 15px 20px;
            border-radius: 6px;
            font-weight: 500;
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            max-width: 500px;
            opacity: 0;
            transition: opacity 0.3s;
        `;
        
        // タイプ別スタイル
        if (type === 'success') {
            alert.style.background = '#d4edda';
            alert.style.color = '#155724';
            alert.style.border = '1px solid #c3e6cb';
        } else if (type === 'error') {
            alert.style.background = '#f8d7da';
            alert.style.color = '#721c24';
            alert.style.border = '1px solid #f5c6cb';
        } else if (type === 'warning') {
            alert.style.background = '#fff3cd';
            alert.style.color = '#856404';
            alert.style.border = '1px solid #ffeaa7';
        }
        
        document.body.appendChild(alert);
        
        setTimeout(() => {
            alert.style.opacity = '1';
        }, 10);
        
        setTimeout(() => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.remove();
            }, 300);
        }, duration);
    },
    
    // ローディング状態の制御
    setLoading: function(element, loading) {
        if (loading) {
            element.disabled = true;
            element.style.opacity = '0.6';
            element.style.cursor = 'not-allowed';
        } else {
            element.disabled = false;
            element.style.opacity = '';
            element.style.cursor = '';
        }
    }
};

// グローバルに公開
window.adminUtils = adminUtils;

// 編集機能の初期化
function initEditFeatures() {
    // 編集ボタンのイベントリスナーを設定
    document.addEventListener('click', function(e) {
        // 編集ボタンがクリックされた場合
        if (e.target.textContent && e.target.textContent.includes('編集')) {
            const button = e.target;
            const row = button.closest('tr');
            
            if (row) {
                const id = parseInt(row.dataset.id);
                const path = window.location.pathname;
                
                // ページ別の編集処理
                if (path.includes('/admin/news') && typeof editNews === 'function') {
                    editNews(id);
                } else if (path.includes('/admin/participants') && typeof editParticipant === 'function') {
                    editParticipant(id);
                } else if (path.includes('/admin/schedules') && typeof editSchedule === 'function') {
                    editSchedule(id);
                } else if (path.includes('/admin/information') && typeof editInfo === 'function') {
                    editInfo(id);
                }
            }
        }
        
        // 削除ボタンがクリックされた場合
        if (e.target.textContent && e.target.textContent.includes('削除')) {
            e.preventDefault(); // onclickイベントをキャンセル
            const button = e.target;
            const row = button.closest('tr');
            
            if (row) {
                const id = parseInt(row.dataset.id);
                const path = window.location.pathname;
                
                // 統一削除処理を呼び出し
                handleDelete(id, path);
            }
        }
    });
}

// 統一削除処理
function handleDelete(id, path) {
    let itemType = '';
    let confirmMessage = '';
    let apiEndpoint = '';
    
    if (path.includes('/admin/news')) {
        itemType = 'お知らせ';
        confirmMessage = 'このお知らせを削除しますか？';
        apiEndpoint = `/admin/news/${id}`;
    } else if (path.includes('/admin/participants')) {
        itemType = '参加者';
        confirmMessage = 'この参加者を削除しますか？関連するプロフィール情報も削除されます。';
        apiEndpoint = `/admin/participants/${id}`;
    } else if (path.includes('/admin/schedules')) {
        itemType = '行程';
        confirmMessage = 'この行程を削除しますか？';
        apiEndpoint = `/admin/schedules/${id}`;
    } else if (path.includes('/admin/information')) {
        itemType = '必要情報';
        confirmMessage = 'この必要情報を削除しますか？';
        apiEndpoint = `/admin/information/${id}`;
    }
    
    if (itemType && confirm(confirmMessage)) {
        console.log(`🗑️ 削除処理開始 - ${itemType} ID:${id}`);
        
        fetch(apiEndpoint, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
            }
        })
        .then(response => {
            console.log(`📡 削除API応答 - Status: ${response.status}`);
            return response.json();
        })
        .then(data => {
            console.log(`📊 削除API結果:`, data);
            if (data.success) {
                adminUtils.showAlert('success', `${itemType}を削除しました`);
                setTimeout(() => {
                    location.reload();
                }, 1000);
            } else {
                adminUtils.showAlert('error', `削除に失敗しました: ${data.error || 'Unknown error'}`);
            }
        })
        .catch(error => {
            console.error(`❌ 削除エラー:`, error);
            adminUtils.showAlert('error', '削除処理でエラーが発生しました');
        });
    }
}

// 一括削除機能の初期化
function initBulkDeleteFeatures() {
    // データがあるページでのみ一括削除UIを表示
    const itemCheckboxes = document.querySelectorAll('.item-checkbox');
    const bulkActions = document.querySelector('.bulk-actions');
    
    if (itemCheckboxes.length > 0 && bulkActions) {
        bulkActions.style.display = 'flex';
        
        // 全選択チェックボックスのイベント
        const selectAll = document.getElementById('selectAll');
        const selectAllInTable = document.getElementById('selectAllInTable');
        
        if (selectAll) {
            selectAll.addEventListener('change', function() {
                toggleAllCheckboxes(this.checked);
                updateBulkDeleteButton();
            });
        }
        
        if (selectAllInTable) {
            selectAllInTable.addEventListener('change', function() {
                toggleAllCheckboxes(this.checked);
                updateBulkDeleteButton();
                
                // 上部の全選択チェックボックスと同期
                if (selectAll) {
                    selectAll.checked = this.checked;
                }
            });
        }
        
        // 個別チェックボックスのイベント
        itemCheckboxes.forEach(checkbox => {
            checkbox.addEventListener('change', updateBulkDeleteButton);
        });
        
        // 一括削除ボタンのイベント
        const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
        if (bulkDeleteBtn) {
            bulkDeleteBtn.addEventListener('click', handleBulkDelete);
        }
    }
}

// 全チェックボックスの切り替え
function toggleAllCheckboxes(checked) {
    const itemCheckboxes = document.querySelectorAll('.item-checkbox');
    itemCheckboxes.forEach(checkbox => {
        checkbox.checked = checked;
    });
}

// 一括削除ボタンの状態更新
function updateBulkDeleteButton() {
    const checkedBoxes = document.querySelectorAll('.item-checkbox:checked');
    const count = checkedBoxes.length;
    const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
    const selectedCount = document.getElementById('selectedCount');
    
    if (selectedCount) {
        selectedCount.textContent = count;
    }
    
    if (bulkDeleteBtn) {
        bulkDeleteBtn.disabled = count === 0;
    }
    
    // 全選択チェックボックスの状態更新
    const allCheckboxes = document.querySelectorAll('.item-checkbox');
    const selectAllCheckbox = document.getElementById('selectAll');
    const selectAllInTableCheckbox = document.getElementById('selectAllInTable');
    
    if (selectAllCheckbox) {
        if (count === 0) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = false;
        } else if (count === allCheckboxes.length) {
            selectAllCheckbox.indeterminate = false;
            selectAllCheckbox.checked = true;
        } else {
            selectAllCheckbox.indeterminate = true;
            selectAllCheckbox.checked = false;
        }
    }
    
    if (selectAllInTableCheckbox) {
        if (count === 0) {
            selectAllInTableCheckbox.indeterminate = false;
            selectAllInTableCheckbox.checked = false;
        } else if (count === allCheckboxes.length) {
            selectAllInTableCheckbox.indeterminate = false;
            selectAllInTableCheckbox.checked = true;
        } else {
            selectAllInTableCheckbox.indeterminate = true;
            selectAllInTableCheckbox.checked = false;
        }
    }
}

// 一括削除の処理
function handleBulkDelete() {
    const checkedBoxes = document.querySelectorAll('.item-checkbox:checked');
    const ids = Array.from(checkedBoxes).map(checkbox => checkbox.value);
    
    if (ids.length === 0) {
        adminUtils.showAlert('warning', '削除するアイテムを選択してください');
        return;
    }
    
    // ページに応じてタイプを決定
    const path = window.location.pathname;
    let type = '';
    let itemName = '';
    
    if (path.includes('/admin/news')) {
        type = 'news';
        itemName = 'お知らせ';
    } else if (path.includes('/admin/schedules')) {
        type = 'schedules';
        itemName = '行程';
    } else if (path.includes('/admin/participants')) {
        type = 'participants';
        itemName = '参加者';
    } else if (path.includes('/admin/information')) {
        type = 'information';
        itemName = '必要情報';
    }
    
    const confirmMessage = `選択した${ids.length}件の${itemName}を削除しますか？\nこの操作は取り消せません。`;
    
    if (confirm(confirmMessage)) {
        bulkDelete(type, ids, itemName);
    }
}

// 一括削除API呼び出し
function bulkDelete(type, ids, itemName) {
    const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
    const originalText = bulkDeleteBtn.innerHTML;
    
    // ローディング表示
    bulkDeleteBtn.innerHTML = '削除中...';
    bulkDeleteBtn.disabled = true;
    
    console.log(`🗑️ 一括削除開始 - Type: ${type}, Count: ${ids.length}`);
    
    fetch('/admin/api/bulk-delete', {
        method: 'DELETE',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            type: type,
            ids: ids
        })
    })
    .then(response => {
        console.log(`📡 一括削除API応答 - Status: ${response.status}`);
        return response.json();
    })
    .then(data => {
        console.log(`📊 一括削除API結果:`, data);
        if (data.success) {
            adminUtils.showAlert('success', `${data.deletedCount}件の${itemName}を削除しました`);
            setTimeout(() => {
                location.reload();
            }, 1500);
        } else {
            adminUtils.showAlert('error', `削除に失敗しました: ${data.message || data.error || 'Unknown error'}`);
        }
    })
    .catch(error => {
        console.error(`❌ 一括削除エラー:`, error);
        adminUtils.showAlert('error', '削除処理でエラーが発生しました');
    })
    .finally(() => {
        // ボタンを元に戻す
        bulkDeleteBtn.innerHTML = originalText;
        if (document.querySelectorAll('.item-checkbox:checked').length === 0) {
            bulkDeleteBtn.disabled = true;
        }
    });
}