# fix013.md - 学生プロフィール編集認証強化 & G班追加

## 📋 修正概要
**作成日**: 2025-10-29  
**優先度**: 🔴 高（本番環境移行前の必須修正）  
**カテゴリ**: セキュリティ修正 + 機能追加

---

## 🚨 修正が必要な問題

### 問題1: 学生プロフィール編集の認証不備
**現状の問題**:
- 学生の詳細画面（`/student/:id`）から編集ボタンを経由すると、ログインなしでプロフィール編集ページにアクセスできてしまう
- 編集ページ（`/student/:id/edit`）への直接アクセス時の認証が不十分

**セキュリティリスク**:
- 未認証ユーザーによる学生プロフィールの不正改ざん
- 他の学生のプロフィールを勝手に編集される可能性

**期待される動作**:
- 未認証ユーザーは編集ボタンが表示されない
- 編集ボタンをクリックした際、未ログインであればログインページへリダイレクト
- 編集ページへの直接アクセス時も認証チェックを実施
- 本人または管理者以外はアクセス不可（403エラー）

### 問題2: 学生の班がG班まで対応していない
**現状の問題**:
- 現在の班はA班～F班までの6班のみ
- G班の学生を追加できない

**必要な対応**:
- 管理画面の参加者追加/編集フォームでG班を選択可能にする
- フロントエンド（メインページ）でG班を表示できるようにする

---

## ✅ 修正内容

### 修正1: 学生プロフィール編集の認証強化

#### 1-1. `routes/student.js` の修正

**ファイル**: `routes/student.js`

**修正箇所1**: `requireStudentAuth` ミドルウェアの強化
```javascript
// 既存のrequireStudentAuthミドルウェアを以下のように修正

function requireStudentAuth(req, res, next) {
  const studentId = parseInt(req.params.id);
  
  // 管理者の場合は全学生のプロフィール編集可能
  if (req.session.isAdmin) {
    return next();
  }
  
  // 学生本人の場合のみ編集可能
  if (req.session.studentId && req.session.studentId === studentId) {
    return next();
  }
  
  // 未認証の場合はログインページへリダイレクト（編集ページのURLを保持）
  return res.redirect(`/student/login?redirect=/student/${studentId}/edit`);
}
```

**修正箇所2**: プロフィール表示ページの権限チェック強化
```javascript
// GET /student/:id - プロフィール表示ページ
router.get('/:id', (req, res) => {
  const studentId = parseInt(req.params.id);
  
  db.get(`
    SELECT p.*, s.participation_comment, s.hobby_selfpr, s.prestudy_task 
    FROM participants p
    LEFT JOIN student_profiles s ON p.id = s.student_id
    WHERE p.id = ? AND p.type = 'student'
  `, [studentId], (err, student) => {
    if (err) {
      return res.status(500).render('error', { 
        message: 'データベースエラーが発生しました',
        error: err 
      });
    }
    
    if (!student) {
      return res.status(404).render('error', { 
        message: '学生が見つかりません',
        error: { status: 404 }
      });
    }
    
    // 編集権限の判定
    const canEdit = req.session.isAdmin || 
                   (req.session.studentId && req.session.studentId === studentId);
    
    // ログイン状態の判定
    const isLoggedIn = req.session.isAdmin || req.session.studentId;
    
    res.render('student-profile', { 
      student, 
      canEdit,
      isLoggedIn,
      currentStudentId: req.session.studentId || null
    });
  });
});
```

**修正箇所3**: 編集ページと更新処理に認証ミドルウェアを適用（確認）
```javascript
// GET /student/:id/edit - 編集ページ（認証必須）
router.get('/:id/edit', requireStudentAuth, (req, res) => {
  // 既存のコード
});

// POST /student/:id/update - 更新処理（認証必須）
router.post('/:id/update', requireStudentAuth, (req, res) => {
  // 既存のコード
});
```

#### 1-2. `views/student-profile.ejs` の修正

**ファイル**: `views/student-profile.ejs`

**修正箇所**: 編集ボタンの条件分岐を厳密化
```ejs
<!-- 既存のボタン部分を以下のように修正 -->

<div class="action-buttons">
  <% if (typeof canEdit !== 'undefined' && canEdit) { %>
    <!-- 編集権限がある場合（本人ログイン済みまたは管理者） -->
    <a href="/student/<%= student.id %>/edit" class="btn btn-primary">
      <i class="bi bi-pencil"></i> プロフィールを編集
    </a>
    
    <% if (typeof isLoggedIn !== 'undefined' && isLoggedIn && !req.session.isAdmin) { %>
      <!-- 学生本人がログインしている場合のみログアウトボタン表示 -->
      <a href="/student/logout" class="btn btn-secondary">
        <i class="bi bi-box-arrow-right"></i> ログアウト
      </a>
    <% } %>
  <% } else { %>
    <!-- 編集権限がない場合 -->
    <a href="/student/login?redirect=/student/<%= student.id %>/edit" class="btn btn-primary">
      <i class="bi bi-box-arrow-in-right"></i> ログインして編集
    </a>
  <% } %>
  
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-arrow-left"></i> トップページに戻る
  </a>
</div>
```

#### 1-3. `views/student/login.ejs` の修正確認

**ファイル**: `views/student/login.ejs`

**確認事項**: redirect パラメータが正しく処理されているか確認
```ejs
<!-- フォーム内に以下が含まれているか確認 -->
<input type="hidden" name="redirect" value="<%= typeof redirect !== 'undefined' ? redirect : '' %>">
```

---

### 修正2: G班の追加

#### 2-1. `views/admin/participants.ejs` の修正

**ファイル**: `views/admin/participants.ejs`

**修正箇所**: 班の選択肢にG班を追加
```html
<!-- 参加者追加/編集フォームの班選択部分を修正 -->

<div class="form-group">
  <label for="group">班 <span style="color: red;">*</span></label>
  <select id="group" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>
```

**編集モーダルの班選択も同様に修正**:
```html
<!-- 編集モーダル内の班選択も同じようにG班を追加 -->
<div class="form-group">
  <label for="editGroup">班 <span style="color: red;">*</span></label>
  <select id="editGroup" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>
```

#### 2-2. `routes/index.js` の修正

**ファイル**: `routes/index.js`

**修正箇所**: G班のデータを取得・表示できるように修正
```javascript
// メインページのルーティング内で、班の配列にGを追加

router.get('/', (req, res) => {
  // ... 既存のコード ...
  
  // 学生を班別にグループ化（A～Gまで対応）
  const groups = ['A', 'B', 'C', 'D', 'E', 'F', 'G'];  // ← Gを追加
  const groupedStudents = {};
  
  groups.forEach(group => {
    groupedStudents[group] = students.filter(s => s.group === group);
  });
  
  // ... 残りの既存コード ...
});
```

#### 2-3. `views/index.ejs` の修正（必要に応じて）

**ファイル**: `views/index.ejs`

**確認事項**: G班が自動的に表示されるか確認
- 現在のコードが動的に班を表示している場合は修正不要
- もし班が固定で記述されている場合は、G班のセクションを追加

**修正例（もし固定記述の場合）**:
```ejs
<!-- 学生セクション内で班ごとに表示している場合 -->

<% ['A', 'B', 'C', 'D', 'E', 'F', 'G'].forEach(group => { %>
  <% if (groupedStudents[group] && groupedStudents[group].length > 0) { %>
    <div class="group-section">
      <h3 class="group-title"><%= group %>班</h3>
      <div class="participants-grid">
        <% groupedStudents[group].forEach(student => { %>
          <!-- 学生カード表示 -->
        <% }) %>
      </div>
    </div>
  <% } %>
<% }) %>
```

---

## 🧪 動作確認テスト

### セキュリティ修正のテスト

#### テスト1: 未ログイン状態での編集ボタン確認
1. ブラウザのシークレットモードで `http://localhost:3000/student/1` にアクセス
2. **期待結果**: 「ログインして編集」ボタンが表示される
3. **期待結果**: 「プロフィールを編集」ボタンは表示されない

#### テスト2: 未ログイン状態での編集ページ直接アクセス
1. シークレットモードで `http://localhost:3000/student/1/edit` に直接アクセス
2. **期待結果**: ログインページ(`/student/login?redirect=/student/1/edit`)にリダイレクトされる
3. **期待結果**: 編集ページは表示されない

#### テスト3: ログイン後の編集ボタン確認
1. 学生1としてログイン（パスワード: password1など）
2. 自分のプロフィールページ `http://localhost:3000/student/1` にアクセス
3. **期待結果**: 「プロフィールを編集」ボタンが表示される
4. **期待結果**: 「ログアウト」ボタンが表示される

#### テスト4: 他の学生のプロフィール編集試行
1. 学生1としてログイン
2. 学生2の編集ページ `http://localhost:3000/student/2/edit` にアクセス
3. **期待結果**: 403エラーまたはログインページにリダイレクト

#### テスト5: 管理者権限での編集確認
1. 管理者としてログイン（admin / dev2590）
2. 任意の学生の編集ページ `http://localhost:3000/student/1/edit` にアクセス
3. **期待結果**: 編集ページが表示され、編集可能

### G班追加のテスト

#### テスト6: 管理画面でG班の参加者を追加
1. 管理画面（`http://localhost:3000/admin`）にログイン
2. 参加者管理ページで「参加者を追加」をクリック
3. 班の選択肢に「G班」があることを確認
4. G班の学生を追加してみる
5. **期待結果**: G班の学生が正常に追加される

#### テスト7: メインページでG班が表示されるか確認
1. メインページ（`http://localhost:3000`）にアクセス
2. 参加者セクションまでスクロール
3. **期待結果**: 「G班」のセクションが表示される
4. **期待結果**: G班の学生が正しく表示される

#### テスト8: G班の学生を編集
1. 管理画面で既存のG班学生の編集をクリック
2. 班の選択肢で「G班」が選択されていることを確認
3. **期待結果**: 編集フォームが正しく動作する

---

## 📁 修正対象ファイル一覧

### セキュリティ修正
- [ ] `routes/student.js` - 認証ミドルウェアの強化、権限チェック追加
- [ ] `views/student-profile.ejs` - 編集ボタンの条件分岐厳密化
- [ ] `views/student/login.ejs` - redirect パラメータの確認

### G班追加
- [ ] `views/admin/participants.ejs` - 班選択肢にG班追加
- [ ] `routes/index.js` - G班のデータ取得・表示対応
- [ ] `views/index.ejs` - G班の表示確認（必要に応じて修正）

---

## 🔧 実装手順

### ステップ1: セキュリティ修正（最優先）
1. `routes/student.js` の `requireStudentAuth` ミドルウェアを修正
2. 同じファイル内のプロフィール表示ページ（GET /student/:id）の権限チェック追加
3. `views/student-profile.ejs` の編集ボタン条件分岐を修正
4. サーバーを再起動してテスト実施

### ステップ2: G班追加
1. `views/admin/participants.ejs` の班選択肢を修正
2. `routes/index.js` の班配列にGを追加
3. `views/index.ejs` の班表示を確認（必要に応じて修正）
4. サーバーを再起動してテスト実施

### ステップ3: 総合テスト
1. 上記の全テストケースを実施
2. 問題があれば修正
3. 本番環境への移行準備

---

## ⚠️ 注意事項

### セキュリティ関連
- **必ず未ログイン状態でのテストを実施してください**
- ブラウザのシークレットモードを使用してテストすることを推奨
- セッションが残っている場合、正しくテストできないため、ブラウザを完全に閉じてから再度テストしてください

### データベース関連
- G班の追加はデータベーススキーマの変更は不要（groupフィールドは文字列型）
- 既存のA～F班のデータには影響なし

### 後方互換性
- 既存の学生ログイン機能への影響なし
- 管理画面の既存機能への影響なし
- 既存のA～F班のデータはそのまま表示される

---

## 📋 完了チェックリスト

### セキュリティ修正
- [ ] `requireStudentAuth` ミドルウェアの修正完了
- [ ] プロフィール表示ページの権限チェック追加完了
- [ ] `student-profile.ejs` の条件分岐修正完了
- [ ] テスト1～5の全てがパス

### G班追加
- [ ] 管理画面の班選択肢にG班追加完了
- [ ] メインページの班配列にG追加完了
- [ ] テスト6～8の全てがパス

### 総合確認
- [ ] サーバーがエラーなく起動
- [ ] 管理画面が正常動作
- [ ] メインページが正常表示
- [ ] 学生ログイン機能が正常動作
- [ ] 本番環境移行準備完了

---

## 🎯 期待される結果

### セキュリティ
- 未認証ユーザーは学生プロフィールを編集できない
- 学生本人のみが自分のプロフィールを編集できる
- 管理者は全学生のプロフィールを編集できる
- 編集ページへの直接アクセスも適切に制御される

### 機能
- G班の学生を管理画面で追加・編集できる
- メインページでG班が正しく表示される
- A～F班の既存機能には影響がない

---

**修正完了後、このファイルを `history.md` に追記してください。**
