# fix013.md - 学生プロフィール編集仕様変更 & G班追加（修正版）

## 📋 修正概要
**作成日**: 2025-10-29  
**優先度**: 🔴 高（本番環境移行前の必須修正）  
**カテゴリ**: 仕様変更 + 機能追加

---

## 🚨 修正が必要な問題

### 問題1: 学生プロフィール編集の仕様変更
**現状の問題**:
- 参加者詳細画面（`/student/:id`）からの編集ボタン経由でエラーが発生
- 「学生が見つかりません」エラーが表示される

**新しい仕様**:
- **参加者詳細画面からの編集機能は完全に削除**
- メインページ下部の「学生ログイン」フォームからのみログイン・編集可能にする
- ログイン後は自動的に自分の編集ページにリダイレクト
- 参加者詳細画面は純粋な「閲覧専用」ページとする

**この変更のメリット**:
- シンプルな導線で混乱を防ぐ
- エラーの発生を防止
- セキュリティリスクを低減

### 問題2: 学生の班がG班まで対応していない
**現状の問題**:
- 現在の班はA班～F班までの6班のみ
- G班の学生を追加できない

**必要な対応**:
- 管理画面の参加者追加/編集フォームでG班を選択可能にする
- フロントエンド（メインページ）でG班を表示できるようにする

---

## ✅ 修正内容

### 修正1: 学生プロフィール編集の仕様変更

#### 1-1. `routes/student.js` の修正

**ファイル**: `routes/student.js`

**修正箇所1**: プロフィール表示ページ（GET /student/:id）をシンプルに変更
```javascript
// GET /student/:id - プロフィール表示ページ（閲覧専用）
router.get('/:id', (req, res) => {
  const studentId = parseInt(req.params.id);
  
  db.get(`
    SELECT p.*, s.participation_comment, s.hobby_selfpr, s.prestudy_task 
    FROM participants p
    LEFT JOIN student_profiles s ON p.id = s.student_id
    WHERE p.id = ? AND p.type = 'student'
  `, [studentId], (err, student) => {
    if (err) {
      return res.status(500).render('error', { 
        message: 'データベースエラーが発生しました',
        error: err 
      });
    }
    
    if (!student) {
      return res.status(404).render('error', { 
        message: '学生が見つかりません',
        error: { status: 404 }
      });
    }
    
    // 閲覧専用ページなので、編集関連の変数は渡さない
    res.render('student-profile', { 
      student
    });
  });
});
```

**修正箇所2**: `requireStudentAuth` ミドルウェアの確認
```javascript
// このミドルウェアが存在することを確認（変更不要）
function requireStudentAuth(req, res, next) {
  const studentId = parseInt(req.params.id);
  
  // 管理者の場合は全学生のプロフィール編集可能
  if (req.session.isAdmin) {
    return next();
  }
  
  // 学生本人の場合のみ編集可能
  if (req.session.studentId && req.session.studentId === studentId) {
    return next();
  }
  
  // 未認証の場合はメインページへリダイレクト
  return res.redirect('/?error=unauthorized');
}
```

**修正箇所3**: 学生ログイン処理の修正（ログイン後に編集ページへリダイレクト）
```javascript
// POST /student/login - ログイン処理
router.post('/login', (req, res) => {
  const { studentId, password } = req.body;
  
  if (!studentId || !password) {
    return res.render('student/login', { 
      error: '学生IDとパスワードを入力してください' 
    });
  }
  
  db.get(
    'SELECT * FROM participants WHERE id = ? AND type = "student"',
    [studentId],
    async (err, student) => {
      if (err) {
        return res.render('student/login', { 
          error: 'ログインに失敗しました' 
        });
      }
      
      if (!student) {
        return res.render('student/login', { 
          error: '学生IDまたはパスワードが間違っています' 
        });
      }
      
      try {
        const match = await bcrypt.compare(password, student.password);
        if (match) {
          req.session.studentId = student.id;
          // ログイン成功後、自分の編集ページへリダイレクト
          return res.redirect(`/student/${student.id}/edit`);
        } else {
          return res.render('student/login', { 
            error: '学生IDまたはパスワードが間違っています' 
          });
        }
      } catch (error) {
        return res.render('student/login', { 
          error: 'ログインに失敗しました' 
        });
      }
    }
  );
});
```

**修正箇所4**: ログアウト処理の確認
```javascript
// GET /student/logout - ログアウト処理（変更不要、念のため確認）
router.get('/logout', (req, res) => {
  // 学生セッション情報をクリア
  req.session.studentId = null;
  // メインページにリダイレクト
  res.redirect('/#student-login');
});
```

**確認事項**: 編集ページと更新処理に`requireStudentAuth`が適用されているか確認
```javascript
// GET /student/:id/edit - 編集ページ（認証必須）
router.get('/:id/edit', requireStudentAuth, (req, res) => {
  // 既存のコード
});

// POST /student/:id/update - 更新処理（認証必須）
router.post('/:id/update', requireStudentAuth, (req, res) => {
  // 既存のコード
});
```

#### 1-2. `views/student-profile.ejs` の修正

**ファイル**: `views/student-profile.ejs`

**修正箇所**: 編集ボタンを完全に削除

編集ボタン部分（アクションボタンエリア）を以下のようにシンプルに変更してください：
```ejs
<!-- アクションボタンエリア - 編集機能を削除 -->
<div class="action-buttons">
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-arrow-left"></i> トップページに戻る
  </a>
</div>

<!-- メインページの学生ログインフォームへの誘導メッセージを追加（任意） -->
<div style="text-align: center; margin-top: 20px; color: #666;">
  <p>※ プロフィールを編集する場合は、<a href="/#student-login">メインページ下部の学生ログイン</a>からログインしてください。</p>
</div>
```

#### 1-3. `views/student/edit.ejs` の修正

**ファイル**: `views/student/edit.ejs`

**修正箇所**: 編集ページに「ログアウト」ボタンと「戻る」ボタンを追加（もしまだなければ）

編集ページの下部（保存ボタンの後）に以下を追加：
```ejs
<!-- 編集ページのアクションボタン -->
<div style="margin-top: 30px; text-align: center;">
  <a href="/student/logout" class="btn btn-secondary">
    <i class="bi bi-box-arrow-right"></i> ログアウト
  </a>
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-house"></i> トップページに戻る
  </a>
</div>
```

#### 1-4. `views/index.ejs` の修正（確認のみ）

**ファイル**: `views/index.ejs`

**確認事項**: メインページ下部に学生ログインフォームが存在することを確認

以下のようなログインフォームがあることを確認してください：
```ejs
<!-- 学生ログインセクション -->
<section id="student-login" class="section">
  <div class="container">
    <h2><i class="bi bi-person-circle"></i> 学生ログイン</h2>
    <form action="/student/login" method="POST">
      <div class="form-group">
        <label for="studentId">学生ID</label>
        <input type="text" id="studentId" name="studentId" required>
      </div>
      <div class="form-group">
        <label for="password">パスワード</label>
        <input type="password" id="password" name="password" required>
      </div>
      <button type="submit" class="btn btn-primary">ログイン</button>
    </form>
  </div>
</section>
```

もしこのセクションがない場合は、ページ下部（フッターの前）に追加してください。

---

### 修正2: G班の追加

#### 2-1. `views/admin/participants.ejs` の修正

**ファイル**: `views/admin/participants.ejs`

**修正箇所**: 班の選択肢にG班を追加

参加者追加フォームと編集モーダルの両方で、以下のように班の選択肢を修正してください：

```html
<!-- 参加者追加フォーム内の班選択 -->
<div class="form-group" id="groupField">
  <label for="group">班 <span style="color: red;">*</span></label>
  <select id="group" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>

<!-- 編集モーダル内の班選択も同様に -->
<div class="form-group" id="editGroupField">
  <label for="editGroup">班 <span style="color: red;">*</span></label>
  <select id="editGroup" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>
```

#### 2-2. `routes/index.js` の修正

**ファイル**: `routes/index.js`

**修正箇所**: G班のデータを取得・表示できるように修正

メインページのルーティング（GET /）内で、班の配列にGを追加してください：

```javascript
router.get('/', (req, res) => {
  // ... 既存のコード（お知らせ、行程表などの取得） ...
  
  // 学生を班別にグループ化（A～Gまで対応）
  const groups = ['A', 'B', 'C', 'D', 'E', 'F', 'G'];  // ← Gを追加
  const groupedStudents = {};
  
  groups.forEach(group => {
    groupedStudents[group] = students.filter(s => s.group === group);
  });
  
  // ... 残りの既存コード（render部分） ...
});
```

#### 2-3. `views/index.ejs` の修正（必要に応じて）

**ファイル**: `views/index.ejs`

**確認事項**: G班が自動的に表示されるか確認
- 現在のコードが動的に班を表示している場合は修正不要
- もし班が固定で記述されている場合は、G班のセクションを追加

**修正例（もし固定記述の場合）**:
```ejs
<!-- 学生セクション内で班ごとに表示している場合 -->

<% ['A', 'B', 'C', 'D', 'E', 'F', 'G'].forEach(group => { %>
  <% if (groupedStudents[group] && groupedStudents[group].length > 0) { %>
    <div class="group-section">
      <h3 class="group-title"><%= group %>班</h3>
      <div class="participants-grid">
        <% groupedStudents[group].forEach(student => { %>
          <!-- 学生カード表示 -->
        <% }) %>
      </div>
    </div>
  <% } %>
<% }) %>
```

---

## 🧪 動作確認テスト

### 仕様変更のテスト

#### テスト1: 参加者詳細画面に編集ボタンがないことを確認
1. メインページ（`http://localhost:3000`）から任意の学生をクリック
2. 参加者詳細画面（`http://localhost:3000/student/1`など）が表示される
3. **期待結果**: 「プロフィールを編集」ボタンや「ログインして編集」ボタンが表示されない
4. **期待結果**: 「トップページに戻る」ボタンのみ表示される

#### テスト2: メインページから学生ログインできることを確認
1. メインページ下部の「学生ログイン」セクションまでスクロール
2. 学生ID（例: 1）とパスワードを入力してログイン
3. **期待結果**: ログイン成功後、自動的に編集ページ（`/student/1/edit`）にリダイレクトされる

#### テスト3: 編集ページで編集できることを確認
1. 学生としてログイン後、編集ページが表示される
2. 各項目（研修参加コメント、趣味PR、事前学習課題）を編集
3. 「保存」ボタンをクリック
4. **期待結果**: 変更が正常に保存される
5. **期待結果**: 保存後、プロフィールページまたはメインページにリダイレクトされる

#### テスト4: 未ログイン状態で編集ページに直接アクセスできないことを確認
1. ブラウザのシークレットモードで `http://localhost:3000/student/1/edit` に直接アクセス
2. **期待結果**: メインページまたはエラーページにリダイレクトされる
3. **期待結果**: 編集ページは表示されない

#### テスト5: 他の学生の編集ページにアクセスできないことを確認
1. 学生1としてログイン
2. 学生2の編集ページ（`http://localhost:3000/student/2/edit`）にアクセスを試みる
3. **期待結果**: アクセスが拒否される（メインページにリダイレクトまたはエラー表示）

#### テスト6: ログアウト後、編集ページにアクセスできないことを確認
1. 学生としてログイン
2. 編集ページの「ログアウト」ボタンをクリック
3. ブラウザの「戻る」ボタンで編集ページに戻ろうとする
4. **期待結果**: 編集ページにアクセスできず、メインページにリダイレクトされる

#### テスト7: 管理者は引き続き編集できることを確認
1. 管理者としてログイン（`http://localhost:3000/admin`）
2. 管理画面から任意の学生のプロフィールを編集
3. **期待結果**: 管理画面から全ての学生の編集が可能

### G班追加のテスト

#### テスト8: 管理画面でG班の参加者を追加
1. 管理画面（`http://localhost:3000/admin`）にログイン
2. 参加者管理ページで「参加者を追加」をクリック
3. 班の選択肢に「G班」があることを確認
4. テストでG班の学生を1人追加
5. **期待結果**: G班の学生が正常に追加される

#### テスト9: メインページでG班が表示されるか確認
```bash
http://localhost:3000
```
- **期待結果**: 参加者セクションに「G班」が表示される
- **期待結果**: G班の学生が正しく表示される

#### テスト10: G班の学生を編集
1. 管理画面でG班学生の編集をクリック
2. 班の選択肢で「G班」が選択されていることを確認
3. **期待結果**: 編集フォームが正しく動作する

---

## 📁 修正対象ファイル一覧

### 仕様変更
- [ ] `routes/student.js` - プロフィール表示ページの簡素化、ログイン処理の修正
- [ ] `views/student-profile.ejs` - 編集ボタンの完全削除
- [ ] `views/student/edit.ejs` - ログアウトボタンの追加（必要に応じて）
- [ ] `views/index.ejs` - 学生ログインフォームの確認

### G班追加
- [ ] `views/admin/participants.ejs` - 班選択肢にG班追加
- [ ] `routes/index.js` - G班のデータ取得・表示対応
- [ ] `views/index.ejs` - G班の表示確認（必要に応じて修正）

---

## 🔧 実装手順

### ステップ1: 仕様変更（最優先）
1. `routes/student.js` のプロフィール表示ページを修正（編集権限の判定を削除）
2. 同じファイル内のログイン処理を修正（ログイン後に編集ページへリダイレクト）
3. `views/student-profile.ejs` から編集ボタンを完全削除
4. `views/student/edit.ejs` にログアウトボタンを追加（もしなければ）
5. サーバーを再起動してテスト実施（テスト1～7）

### ステップ2: G班追加
1. `views/admin/participants.ejs` の班選択肢を修正
2. `routes/index.js` の班配列にGを追加
3. `views/index.ejs` の班表示を確認（必要に応じて修正）
4. サーバーを再起動してテスト実施（テスト8～10）

### ステップ3: 総合テスト
1. 上記の全テストケースを実施
2. 問題があれば修正
3. 本番環境への移行準備

---

## ⚠️ 注意事項

### 仕様変更関連
- **参加者詳細画面は完全に閲覧専用になります**
- 編集したい学生は必ずメインページ下部のログインフォームからログインします
- この変更により、導線がシンプルになり、エラーが発生しにくくなります

### データベース関連
- G班の追加はデータベーススキーマの変更は不要（groupフィールドは文字列型）
- 既存のA～F班のデータには影響なし

### 後方互換性
- 既存の学生ログイン機能への影響なし（むしろ改善）
- 管理画面の既存機能への影響なし
- 既存のA～F班のデータはそのまま表示される

### セキュリティ
- 編集ページへの直接アクセスは引き続き認証必須
- 本人または管理者のみが編集可能（変更なし）
- 未認証でのアクセスは適切にブロックされる

---

## 📋 完了チェックリスト

### 仕様変更
- [ ] `routes/student.js`のプロフィール表示ページ修正完了
- [ ] ログイン処理の修正完了（ログイン後に編集ページへリダイレクト）
- [ ] `student-profile.ejs`の編集ボタン削除完了
- [ ] `student/edit.ejs`のログアウトボタン追加完了
- [ ] テスト1～7の全てがパス

### G班追加
- [ ] 管理画面の班選択肢にG班追加完了
- [ ] メインページの班配列にG追加完了
- [ ] `index.ejs`の確認完了
- [ ] テスト8～10の全てがパス

### 総合確認
- [ ] サーバーがエラーなく起動
- [ ] 管理画面が正常動作
- [ ] メインページが正常表示
- [ ] 学生ログイン機能が正常動作
- [ ] 参加者詳細画面が閲覧専用として正常動作
- [ ] 全てのテストケースがパス

---

## 🎯 期待される結果

### 仕様変更後
- 参加者詳細画面は純粋な閲覧専用ページになる
- 学生はメインページ下部のログインフォームからログインする
- ログイン後は自動的に自分の編集ページに移動する
- 編集後はログアウトして終了（または保存後にメインページに戻る）
- エラーが発生しにくいシンプルな導線

### 機能追加後
- G班の学生を管理画面で追加・編集できる
- メインページでG班が正しく表示される
- A～F班の既存機能には影響がない

---

## 💡 改善点

今回の仕様変更により、以下の改善が期待できます：

1. **シンプルな導線**: 学生は「ログインフォーム」→「編集ページ」という明確な流れ
2. **エラーの削減**: 複雑な権限チェックを削減し、エラーが発生しにくい
3. **セキュリティ**: 編集ページへの直接アクセスは引き続き保護される
4. **ユーザビリティ**: 学生にとってわかりやすい操作方法

---

**修正完了後、このファイルを `history.md` に追記してください。**
