# fix013.md - 学生プロフィール編集仕様変更 & G班追加（最終版）

## 📋 修正概要
**作成日**: 2025-10-29  
**優先度**: 🔴 高（本番環境移行前の必須修正）  
**カテゴリ**: 仕様変更 + 機能追加

---

## 🚨 修正が必要な問題

### 問題1: 学生プロフィール編集の仕様変更
**現状の問題**:
- 参加者詳細画面（`/student/:id`）からの編集ボタン経由でエラーが発生
- 「学生が見つかりません」エラーが表示される
- 学生IDでのログインは学生にとって分かりにくい

**新しい仕様**:
- **参加者詳細画面からの編集機能は完全に削除**
- メインページ下部の「学生ログイン」フォームからのみログイン・編集可能にする
- **ログイン方式を「名前とパスワード」に変更**（学生IDは不要）
- ログイン後は自動的に自分の編集ページにリダイレクト
- 参加者詳細画面は純粋な「閲覧専用」ページとする

**この変更のメリット**:
- シンプルな導線で混乱を防ぐ
- 学生が自分の名前でログインできる（IDを覚える必要なし）
- エラーの発生を防止
- セキュリティリスクを低減

### 問題2: 学生の班がG班まで対応していない
**現状の問題**:
- 現在の班はA班～F班までの6班のみ
- G班の学生を追加できない

**必要な対応**:
- 管理画面の参加者追加/編集フォームでG班を選択可能にする
- フロントエンド（メインページ）でG班を表示できるようにする

---

## ✅ 修正内容

### 修正1: 学生プロフィール編集の仕様変更

#### 1-1. `routes/student.js` の修正

**ファイル**: `routes/student.js`

**修正箇所1**: プロフィール表示ページ（GET /student/:id）をシンプルに変更
```javascript
// GET /student/:id - プロフィール表示ページ（閲覧専用）
router.get('/:id', (req, res) => {
  const studentId = parseInt(req.params.id);
  
  db.get(`
    SELECT p.*, s.participation_comment, s.hobby_selfpr, s.prestudy_task 
    FROM participants p
    LEFT JOIN student_profiles s ON p.id = s.student_id
    WHERE p.id = ? AND p.type = 'student'
  `, [studentId], (err, student) => {
    if (err) {
      return res.status(500).render('error', { 
        message: 'データベースエラーが発生しました',
        error: err 
      });
    }
    
    if (!student) {
      return res.status(404).render('error', { 
        message: '学生が見つかりません',
        error: { status: 404 }
      });
    }
    
    // 閲覧専用ページなので、編集関連の変数は渡さない
    res.render('student-profile', { 
      student
    });
  });
});
```

**修正箇所2**: ログイン処理を修正（名前とパスワードでログイン）
```javascript
// POST /student/login - ログイン処理（名前とパスワード）
router.post('/login', (req, res) => {
  const { name, password } = req.body;
  
  if (!name || !password) {
    return res.render('student/login', { 
      error: '名前とパスワードを入力してください' 
    });
  }
  
  // 名前で学生を検索（スペース除去で柔軟に対応）
  const searchName = name.trim();
  
  db.get(
    'SELECT * FROM participants WHERE name = ? AND type = "student"',
    [searchName],
    async (err, student) => {
      if (err) {
        return res.render('student/login', { 
          error: 'ログインに失敗しました' 
        });
      }
      
      if (!student) {
        return res.render('student/login', { 
          error: '名前またはパスワードが間違っています' 
        });
      }
      
      try {
        const match = await bcrypt.compare(password, student.password);
        if (match) {
          req.session.studentId = student.id;
          // ログイン成功後、自分の編集ページへリダイレクト
          return res.redirect(`/student/${student.id}/edit`);
        } else {
          return res.render('student/login', { 
            error: '名前またはパスワードが間違っています' 
          });
        }
      } catch (error) {
        return res.render('student/login', { 
          error: 'ログインに失敗しました' 
        });
      }
    }
  );
});
```

**修正箇所3**: `requireStudentAuth` ミドルウェアの確認・修正
```javascript
// 既存のrequireStudentAuthを以下のように修正（未認証時のリダイレクト先変更）
function requireStudentAuth(req, res, next) {
  const studentId = parseInt(req.params.id);
  
  // 管理者の場合は全学生のプロフィール編集可能
  if (req.session.isAdmin) {
    return next();
  }
  
  // 学生本人の場合のみ編集可能
  if (req.session.studentId && req.session.studentId === studentId) {
    return next();
  }
  
  // 未認証の場合はメインページの学生ログインセクションへリダイレクト
  return res.redirect('/#student-login');
}
```

**修正箇所4**: ログアウト処理の確認
```javascript
// GET /student/logout - ログアウト処理（変更不要、念のため確認）
router.get('/logout', (req, res) => {
  // 学生セッション情報をクリア
  req.session.studentId = null;
  // メインページの学生ログインセクションにリダイレクト
  res.redirect('/#student-login');
});
```

**確認事項**: 編集ページと更新処理に`requireStudentAuth`が適用されているか確認
```javascript
// GET /student/:id/edit - 編集ページ（認証必須）
router.get('/:id/edit', requireStudentAuth, (req, res) => {
  // 既存のコード
});

// POST /student/:id/update - 更新処理（認証必須）
router.post('/:id/update', requireStudentAuth, (req, res) => {
  // 既存のコード
});
```

#### 1-2. `views/student-profile.ejs` の修正

**ファイル**: `views/student-profile.ejs`

**修正箇所**: 編集ボタンを完全に削除

編集ボタン部分（アクションボタンエリア）を以下のようにシンプルに変更してください：
```ejs
<!-- アクションボタンエリア - 編集機能を削除 -->
<div class="action-buttons" style="text-align: center; margin-top: 30px;">
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-arrow-left"></i> トップページに戻る
  </a>
</div>

<!-- メインページの学生ログインフォームへの誘導メッセージを追加 -->
<div style="text-align: center; margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 8px;">
  <p style="margin: 0; color: #666; font-size: 14px;">
    <i class="bi bi-info-circle"></i> 
    プロフィールを編集する場合は、
    <a href="/#student-login" style="color: #2590ff; font-weight: bold;">メインページ下部の学生ログイン</a>
    から自分の名前とパスワードでログインしてください。
  </p>
</div>
```

#### 1-3. `views/student/login.ejs` の修正

**ファイル**: `views/student/login.ejs`

**修正箇所**: 学生ID → 名前に変更

ログインフォームを以下のように修正してください：
```ejs
<!DOCTYPE html>
<html lang="ja">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>学生ログイン</title>
  <link rel="stylesheet" href="/css/style.css">
</head>
<body>
  <div class="container" style="max-width: 400px; margin: 50px auto; padding: 20px;">
    <h1 style="text-align: center; margin-bottom: 30px;">学生ログイン</h1>
    
    <% if (typeof error !== 'undefined' && error) { %>
      <div class="alert alert-danger" style="padding: 10px; margin-bottom: 20px; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px; color: #721c24;">
        <%= error %>
      </div>
    <% } %>
    
    <form action="/student/login" method="POST">
      <div class="form-group" style="margin-bottom: 20px;">
        <label for="name" style="display: block; margin-bottom: 5px; font-weight: bold;">
          名前 <span style="color: red;">*</span>
        </label>
        <input 
          type="text" 
          id="name" 
          name="name" 
          class="form-control" 
          required 
          placeholder="例: 山田太郎"
          style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
        >
        <small style="display: block; margin-top: 5px; color: #666;">
          登録されている名前を正確に入力してください
        </small>
      </div>
      
      <div class="form-group" style="margin-bottom: 20px;">
        <label for="password" style="display: block; margin-bottom: 5px; font-weight: bold;">
          パスワード <span style="color: red;">*</span>
        </label>
        <input 
          type="password" 
          id="password" 
          name="password" 
          class="form-control" 
          required
          style="width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
        >
      </div>
      
      <button 
        type="submit" 
        class="btn btn-primary" 
        style="width: 100%; padding: 12px; background: #2590ff; color: white; border: none; border-radius: 4px; font-size: 16px; font-weight: bold; cursor: pointer;"
      >
        ログイン
      </button>
    </form>
    
    <div style="text-align: center; margin-top: 20px;">
      <a href="/" style="color: #2590ff; text-decoration: none;">
        <i class="bi bi-arrow-left"></i> トップページに戻る
      </a>
    </div>
  </div>
</body>
</html>
```

**変更点**: 
- `studentId` → `name` に変更
- ラベル「学生ID」→「名前」に変更
- プレースホルダー「例: 1」→「例: 山田太郎」に変更
- 説明文を追加

#### 1-4. `views/student/edit.ejs` の修正

**ファイル**: `views/student/edit.ejs`

**修正箇所**: ログアウトボタンとナビゲーションボタンを追加（もしまだなければ）

編集ページの下部（保存ボタンの後）に以下を追加：
```ejs
<!-- 編集ページのアクションボタン -->
<div style="margin-top: 30px; text-align: center;">
  <button type="submit" form="profileForm" class="btn btn-primary">
    <i class="bi bi-check-circle"></i> 保存
  </button>
  <a href="/student/logout" class="btn btn-secondary">
    <i class="bi bi-box-arrow-right"></i> ログアウト
  </a>
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-house"></i> トップページに戻る
  </a>
</div>
```

**注意**: フォームのid属性が`profileForm`であることを確認してください。違う場合は適宜変更してください。

#### 1-5. `views/index.ejs` の修正

**ファイル**: `views/index.ejs`

**修正箇所**: メインページ下部の学生ログインフォームを名前とパスワード方式に変更

以下のような学生ログインセクションをページ下部（フッターの前）に追加または修正してください：

```ejs
<!-- 学生ログインセクション -->
<section id="student-login" class="section">
  <div class="container">
    <h2 style="text-align: center; margin-bottom: 10px;">
      <i class="bi bi-person-circle"></i> 学生ログイン
    </h2>
    <p style="text-align: center; color: #666; margin-bottom: 30px;">
      プロフィール編集を行う学生の方は、以下からログインしてください。
    </p>
    
    <% if (typeof error !== 'undefined' && error) { %>
      <div class="alert alert-danger" style="max-width: 400px; margin: 0 auto 20px; padding: 10px; background: #f8d7da; border: 1px solid #f5c6cb; border-radius: 4px; color: #721c24; text-align: center;">
        <%= error %>
      </div>
    <% } %>
    
    <form action="/student/login" method="POST" style="max-width: 400px; margin: 0 auto; background: #f8f9fa; padding: 30px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1);">
      <div class="form-group" style="margin-bottom: 20px;">
        <label for="studentName" style="display: block; margin-bottom: 8px; font-weight: bold; color: #333;">
          <i class="bi bi-person"></i> 名前 <span style="color: red;">*</span>
        </label>
        <input 
          type="text" 
          id="studentName" 
          name="name" 
          class="form-control" 
          required 
          placeholder="例: 山田太郎"
          style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
        >
        <small style="display: block; margin-top: 5px; color: #666; font-size: 13px;">
          登録されている名前を正確に入力してください
        </small>
      </div>
      
      <div class="form-group" style="margin-bottom: 25px;">
        <label for="studentPassword" style="display: block; margin-bottom: 8px; font-weight: bold; color: #333;">
          <i class="bi bi-lock"></i> パスワード <span style="color: red;">*</span>
        </label>
        <input 
          type="password" 
          id="studentPassword" 
          name="password" 
          class="form-control" 
          required
          style="width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 4px; font-size: 16px;"
        >
      </div>
      
      <button 
        type="submit" 
        class="btn btn-primary" 
        style="width: 100%; padding: 14px; background: linear-gradient(135deg, #2590ff 0%, #1a73cc 100%); color: white; border: none; border-radius: 4px; font-size: 16px; font-weight: bold; cursor: pointer; transition: transform 0.2s;"
        onmouseover="this.style.transform='translateY(-2px)'"
        onmouseout="this.style.transform='translateY(0)'"
      >
        <i class="bi bi-box-arrow-in-right"></i> ログイン
      </button>
    </form>
    
    <div style="text-align: center; margin-top: 20px; color: #999; font-size: 13px;">
      <p>
        <i class="bi bi-info-circle"></i> 
        名前とパスワードがわからない場合は、引率の先生にお問い合わせください。
      </p>
    </div>
  </div>
</section>
```

**変更点**: 
- `studentId` → `name` に変更
- フィールドラベル「学生ID」→「名前」
- プレースホルダー変更
- デザインを少し改善

---

### 修正2: G班の追加

#### 2-1. `views/admin/participants.ejs` の修正

**ファイル**: `views/admin/participants.ejs`

**修正箇所**: 班の選択肢にG班を追加

参加者追加フォームと編集モーダルの両方で、以下のように班の選択肢を修正してください：

```html
<!-- 参加者追加フォーム内の班選択 -->
<div class="form-group" id="groupField">
  <label for="group">班 <span style="color: red;">*</span></label>
  <select id="group" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>

<!-- 編集モーダル内の班選択も同様に -->
<div class="form-group" id="editGroupField">
  <label for="editGroup">班 <span style="color: red;">*</span></label>
  <select id="editGroup" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>
```

#### 2-2. `routes/index.js` の修正

**ファイル**: `routes/index.js`

**修正箇所**: G班のデータを取得・表示できるように修正

メインページのルーティング（GET /）内で、班の配列にGを追加してください：

```javascript
router.get('/', (req, res) => {
  // ... 既存のコード（お知らせ、行程表などの取得） ...
  
  // 学生を班別にグループ化（A～Gまで対応）
  const groups = ['A', 'B', 'C', 'D', 'E', 'F', 'G'];  // ← Gを追加
  const groupedStudents = {};
  
  groups.forEach(group => {
    groupedStudents[group] = students.filter(s => s.group === group);
  });
  
  // ... 残りの既存コード（render部分） ...
});
```

#### 2-3. `views/index.ejs` の修正（必要に応じて）

**ファイル**: `views/index.ejs`

**確認事項**: G班が自動的に表示されるか確認
- 現在のコードが動的に班を表示している場合は修正不要
- もし班が固定で記述されている場合は、G班のセクションを追加

**修正例（もし固定記述の場合）**:
```ejs
<!-- 学生セクション内で班ごとに表示している場合 -->

<% ['A', 'B', 'C', 'D', 'E', 'F', 'G'].forEach(group => { %>
  <% if (groupedStudents[group] && groupedStudents[group].length > 0) { %>
    <div class="group-section">
      <h3 class="group-title"><%= group %>班</h3>
      <div class="participants-grid">
        <% groupedStudents[group].forEach(student => { %>
          <!-- 学生カード表示 -->
        <% }) %>
      </div>
    </div>
  <% } %>
<% }) %>
```

---

## 🧪 動作確認テスト

### 仕様変更のテスト

#### テスト1: 参加者詳細画面に編集ボタンがないことを確認
1. メインページ（`http://localhost:3000`）から任意の学生をクリック
2. 参加者詳細画面（`http://localhost:3000/student/1`など）が表示される
3. **期待結果**: 「プロフィールを編集」ボタンや「ログインして編集」ボタンが表示されない
4. **期待結果**: 「トップページに戻る」ボタンのみ表示される
5. **期待結果**: ログインへの誘導メッセージが表示される

#### テスト2: メインページから名前とパスワードでログインできることを確認
1. メインページ下部の「学生ログイン」セクションまでスクロール
2. 名前（例: 山田太郎）とパスワードを入力してログイン
3. **期待結果**: ログイン成功後、自動的に編集ページ（`/student/1/edit`）にリダイレクトされる

#### テスト3: 編集ページで編集できることを確認
1. 学生としてログイン後、編集ページが表示される
2. 各項目（研修参加コメント、趣味PR、事前学習課題）を編集
3. 「保存」ボタンをクリック
4. **期待結果**: 変更が正常に保存される
5. **期待結果**: 保存後、プロフィールページまたはメインページにリダイレクトされる

#### テスト4: 間違った名前でログインできないことを確認
1. メインページの学生ログインフォームで存在しない名前を入力
2. **期待結果**: 「名前またはパスワードが間違っています」エラーが表示される
3. **期待結果**: ログインできない

#### テスト5: 未ログイン状態で編集ページに直接アクセスできないことを確認
1. ブラウザのシークレットモードで `http://localhost:3000/student/1/edit` に直接アクセス
2. **期待結果**: メインページの学生ログインセクション（`/#student-login`）にリダイレクトされる
3. **期待結果**: 編集ページは表示されない

#### テスト6: 他の学生の編集ページにアクセスできないことを確認
1. 学生1としてログイン
2. 学生2の編集ページ（`http://localhost:3000/student/2/edit`）にアクセスを試みる
3. **期待結果**: アクセスが拒否される（メインページにリダイレクトまたはエラー表示）

#### テスト7: ログアウト後、編集ページにアクセスできないことを確認
1. 学生としてログイン
2. 編集ページの「ログアウト」ボタンをクリック
3. ブラウザの「戻る」ボタンで編集ページに戻ろうとする
4. **期待結果**: 編集ページにアクセスできず、メインページにリダイレクトされる

#### テスト8: 管理者は引き続き編集できることを確認
1. 管理者としてログイン（`http://localhost:3000/admin`）
2. 管理画面から任意の学生のプロフィールを編集
3. **期待結果**: 管理画面から全ての学生の編集が可能

### G班追加のテスト

#### テスト9: 管理画面でG班の参加者を追加
1. 管理画面（`http://localhost:3000/admin`）にログイン
2. 参加者管理ページで「参加者を追加」をクリック
3. 班の選択肢に「G班」があることを確認
4. テストでG班の学生を1人追加
5. **期待結果**: G班の学生が正常に追加される

#### テスト10: メインページでG班が表示されるか確認
```bash
http://localhost:3000
```
- **期待結果**: 参加者セクションに「G班」が表示される
- **期待結果**: G班の学生が正しく表示される

#### テスト11: G班の学生を編集
1. 管理画面でG班学生の編集をクリック
2. 班の選択肢で「G班」が選択されていることを確認
3. **期待結果**: 編集フォームが正しく動作する

#### テスト12: G班の学生が名前でログインできることを確認
1. メインページの学生ログインフォームでG班の学生の名前とパスワードを入力
2. **期待結果**: 正常にログインできる
3. **期待結果**: 編集ページが表示される

---

## 📁 修正対象ファイル一覧

### 仕様変更（名前とパスワードでのログイン）
- [ ] `routes/student.js` - プロフィール表示・ログイン処理・認証ミドルウェアの修正
- [ ] `views/student-profile.ejs` - 編集ボタンの完全削除
- [ ] `views/student/login.ejs` - 学生ID→名前に変更
- [ ] `views/student/edit.ejs` - ログアウトボタンの追加（必要に応じて）
- [ ] `views/index.ejs` - メインページの学生ログインフォームを名前方式に変更

### G班追加
- [ ] `views/admin/participants.ejs` - 班選択肢にG班追加
- [ ] `routes/index.js` - G班のデータ取得・表示対応
- [ ] `views/index.ejs` - G班の表示確認（必要に応じて修正）

---

## 🔧 実装手順

### ステップ1: 仕様変更（最優先）
1. `routes/student.js` のプロフィール表示ページを修正（編集権限の判定を削除）
2. 同じファイル内のログイン処理を修正（学生ID→名前、編集ページへリダイレクト）
3. 同じファイル内の`requireStudentAuth`ミドルウェアを修正
4. `views/student/login.ejs` を修正（学生ID→名前）
5. `views/student-profile.ejs` から編集ボタンを完全削除
6. `views/student/edit.ejs` にログアウトボタンを追加（もしなければ）
7. `views/index.ejs` のメインページログインフォームを修正（学生ID→名前）
8. サーバーを再起動してテスト実施（テスト1～8）

### ステップ2: G班追加
1. `views/admin/participants.ejs` の班選択肢を修正
2. `routes/index.js` の班配列にGを追加
3. `views/index.ejs` の班表示を確認（必要に応じて修正）
4. サーバーを再起動してテスト実施（テスト9～12）

### ステップ3: 総合テスト
1. 上記の全テストケース（12項目）を実施
2. 問題があれば修正
3. 本番環境への移行準備

---

## ⚠️ 注意事項

### 名前でのログインに関する注意
- **名前は正確に入力する必要があります**（スペース、漢字、ひらがななど）
- データベースに登録されている名前と完全に一致する必要があります
- もし学生が自分の名前を間違えやすい場合は、管理画面で「ログイン用の名前」を確認できる機能を追加することをお勧めします

### 名前の重複に関する注意
- **同姓同名の学生がいる場合、ログインができません**
- この場合は以下の対策が必要です：
  1. 名前に学校名を追加（例: 「山田太郎（○○高校）」）
  2. 別のユニークな識別子を使用
  3. メールアドレスや別のID方式を採用

### データベース関連
- G班の追加はデータベーススキーマの変更は不要（groupフィールドは文字列型）
- 既存のA～F班のデータには影響なし

### 後方互換性
- 既存の学生パスワードは引き続き使用可能
- 管理画面の既存機能への影響なし
- 既存のA～F班のデータはそのまま表示される

### セキュリティ
- 編集ページへの直接アクセスは引き続き認証必須
- 本人または管理者のみが編集可能（変更なし）
- 未認証でのアクセスは適切にブロックされる

---

## 📋 完了チェックリスト

### 仕様変更（名前とパスワードでのログイン）
- [ ] `routes/student.js`のプロフィール表示ページ修正完了
- [ ] ログイン処理の修正完了（名前方式）
- [ ] `requireStudentAuth`ミドルウェアの修正完了
- [ ] `student/login.ejs`の修正完了（名前フィールド）
- [ ] `student-profile.ejs`の編集ボタン削除完了
- [ ] `student/edit.ejs`のログアウトボタン追加完了
- [ ] `index.ejs`のログインフォーム修正完了（名前方式）
- [ ] テスト1～8の全てがパス

### G班追加
- [ ] 管理画面の班選択肢にG班追加完了
- [ ] メインページの班配列にG追加完了
- [ ] `index.ejs`の確認完了
- [ ] テスト9～12の全てがパス

### 総合確認
- [ ] サーバーがエラーなく起動
- [ ] 管理画面が正常動作
- [ ] メインページが正常表示
- [ ] 学生ログイン機能が正常動作（名前方式）
- [ ] 参加者詳細画面が閲覧専用として正常動作
- [ ] 全てのテストケース（12項目）がパス

---

## 🎯 期待される結果

### 仕様変更後
- 参加者詳細画面は純粋な閲覧専用ページになる
- 学生はメインページ下部のログインフォームから**自分の名前**とパスワードでログインする
- ログイン後は自動的に自分の編集ページに移動する
- 編集後はログアウトして終了（または保存後にメインページに戻る）
- エラーが発生しにくいシンプルな導線
- **学生はIDを覚える必要がなく、自分の名前でログインできる**

### 機能追加後
- G班の学生を管理画面で追加・編集できる
- メインページでG班が正しく表示される
- A～F班の既存機能には影響がない

---

## 💡 改善点

今回の仕様変更により、以下の改善が期待できます：

1. **ユーザビリティ向上**: 学生が自分の名前でログインできる（IDを覚える必要なし）
2. **シンプルな導線**: ログインフォーム → 編集ページ という明確な流れ
3. **エラーの削減**: 複雑な権限チェックを削減し、エラーが発生しにくい
4. **セキュリティ維持**: 編集ページへの直接アクセスは引き続き保護される
5. **直感的な操作**: 学生にとってわかりやすい操作方法

---

## 🆘 名前の重複がある場合の対応

もし同姓同名の学生がいる場合は、以下の追加対応が必要です：

### 対応案1: 学校名を含める
```
例: 山田太郎（○○高校） → 山田太郎（東京高校）
```

### 対応案2: ログイン用の別名フィールドを追加
データベースに`login_name`フィールドを追加し、ユニークな名前を設定する

この場合は別途修正が必要ですので、ご相談ください。

---

**修正完了後、このファイルを `history.md` に追記してください。**
