# Claude Code向け 学生ページ認証・権限修正 指示書

## 🚨 セキュリティ問題

**現状:** 学生プロフィールページが誰でも編集可能になっている
**必要な修正:** 本人ログインまたは管理者のみ編集可能にする

## 🎯 修正要件

### 権限設定
1. **学生個別ページ（/student/:id）**
   - 誰でも閲覧可能（変更なし）
   - 「プロフィール編集」ボタンは認証状態でのみ表示

2. **学生編集ページ（/student/:id/edit）**
   - 本人ログイン済み → 編集可能
   - 管理者ログイン済み → 編集可能
   - 未ログインまたは他人 → アクセス拒否

3. **管理画面（/admin）**
   - 管理者のみ全学生の情報編集可能（変更なし）

## 🔧 実装修正

### 1. 学生ログイン機能の追加

#### routes/student.js に認証機能追加
```javascript
const express = require('express');
const router = express.Router();
const bcrypt = require('bcrypt');

// ========================================
// ミドルウェア：学生認証チェック
// ========================================
function requireStudentAuth(req, res, next) {
    // 管理者ログイン済みの場合は許可
    if (req.session && req.session.isAdmin) {
        return next();
    }
    
    // 学生ログイン済みかチェック
    if (req.session && req.session.studentId) {
        const requestedId = parseInt(req.params.id);
        const loggedInId = parseInt(req.session.studentId);
        
        // 本人の場合のみ許可
        if (requestedId === loggedInId) {
            return next();
        }
    }
    
    // 認証されていない場合はログインページへリダイレクト
    return res.redirect(`/student/${req.params.id}/login?redirect=/student/${req.params.id}/edit`);
}

// ========================================
// 学生ログインページ表示
// ========================================
router.get('/:id/login', async (req, res) => {
    try {
        const studentId = req.params.id;
        const redirect = req.query.redirect || `/student/${studentId}`;
        
        // 学生情報取得
        const student = await db.get(
            'SELECT id, name FROM participants WHERE id = ? AND type = "student"',
            [studentId]
        );
        
        if (!student) {
            return res.status(404).render('error', { 
                message: '学生が見つかりません' 
            });
        }
        
        res.render('student-login', { 
            student,
            redirect,
            error: null 
        });
    } catch (error) {
        console.error('Login page error:', error);
        res.status(500).render('error', { 
            message: 'エラーが発生しました' 
        });
    }
});

// ========================================
// 学生ログイン処理
// ========================================
router.post('/:id/login', async (req, res) => {
    try {
        const studentId = parseInt(req.params.id);
        const { password } = req.body;
        const redirect = req.body.redirect || `/student/${studentId}`;
        
        // 学生情報取得
        const student = await db.get(
            'SELECT id, name, password FROM participants WHERE id = ? AND type = "student"',
            [studentId]
        );
        
        if (!student) {
            return res.render('student-login', {
                student: { id: studentId, name: '不明' },
                redirect,
                error: '学生が見つかりません'
            });
        }
        
        // パスワード確認（平文比較 - 簡易実装）
        if (student.password && password === student.password) {
            // セッションに学生ID保存
            req.session.studentId = student.id;
            req.session.studentName = student.name;
            
            // リダイレクト
            return res.redirect(redirect);
        } else {
            // パスワード不一致
            return res.render('student-login', {
                student: { id: student.id, name: student.name },
                redirect,
                error: 'パスワードが正しくありません'
            });
        }
    } catch (error) {
        console.error('Login error:', error);
        res.status(500).render('error', { 
            message: 'ログイン処理でエラーが発生しました' 
        });
    }
});

// ========================================
// 学生ログアウト
// ========================================
router.get('/:id/logout', (req, res) => {
    req.session.studentId = null;
    req.session.studentName = null;
    res.redirect(`/student/${req.params.id}`);
});

// ========================================
// 学生プロフィール表示（変更）
// ========================================
router.get('/:id', async (req, res) => {
    try {
        const studentId = req.params.id;
        
        const student = await db.get(`
            SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
            FROM participants p 
            LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
            WHERE p.id = ? AND p.type = 'student'
        `, [studentId]);
        
        if (!student) {
            return res.status(404).render('error', { 
                message: '学生が見つかりません' 
            });
        }
        
        // 編集権限チェック
        const canEdit = req.session && (
            req.session.isAdmin || 
            (req.session.studentId && parseInt(req.session.studentId) === parseInt(studentId))
        );
        
        res.render('student-profile', { 
            student,
            canEdit, // 編集権限をテンプレートに渡す
            query: req.query 
        });
    } catch (error) {
        console.error('Student profile error:', error);
        res.status(500).render('error', { 
            message: error.message 
        });
    }
});

// ========================================
// 学生編集ページ表示（認証必須）
// ========================================
router.get('/:id/edit', requireStudentAuth, async (req, res) => {
    try {
        const studentId = req.params.id;
        
        const student = await db.get(`
            SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
            FROM participants p 
            LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
            WHERE p.id = ? AND p.type = 'student'
        `, [studentId]);
        
        if (!student) {
            return res.status(404).render('error', { 
                message: '学生が見つかりません' 
            });
        }
        
        res.render('student-edit', { 
            student, 
            error: null,
            isAdmin: req.session.isAdmin || false
        });
    } catch (error) {
        console.error('Student edit page error:', error);
        res.status(500).render('error', { 
            message: 'システムエラーが発生しました' 
        });
    }
});

// ========================================
// 学生プロフィール更新（認証必須）
// ========================================
router.post('/:id/update', requireStudentAuth, async (req, res) => {
    try {
        const studentId = parseInt(req.params.id);
        const { participation_comment, hobby_selfpr, prestudy_task } = req.body;
        
        // データベース更新処理（既存のコード）
        const existingProfile = await db.get(
            'SELECT * FROM student_profiles WHERE participant_id = ?', 
            [studentId]
        );
        
        if (existingProfile) {
            await db.run(`
                UPDATE student_profiles 
                SET participation_comment = ?, 
                    hobby_selfpr = ?, 
                    prestudy_task = ?, 
                    updated_at = CURRENT_TIMESTAMP
                WHERE participant_id = ?
            `, [participation_comment || '', hobby_selfpr || '', prestudy_task || '', studentId]);
        } else {
            await db.run(`
                INSERT INTO student_profiles 
                (participant_id, participation_comment, hobby_selfpr, prestudy_task, updated_at)
                VALUES (?, ?, ?, ?, CURRENT_TIMESTAMP)
            `, [studentId, participation_comment || '', hobby_selfpr || '', prestudy_task || '']);
        }
        
        res.redirect(`/student/${studentId}?updated=true`);
        
    } catch (error) {
        console.error('Student update error:', error);
        res.status(500).render('error', { 
            message: 'プロフィールの更新に失敗しました' 
        });
    }
});

module.exports = router;
```

### 2. 学生ログインページテンプレート

#### views/student-login.ejs（新規作成）
```html
<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ログイン - <%= student.name %> | 2590地区インターアクト研修旅行</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.7.2/font/bootstrap-icons.css">
    <link href="/css/style.css" rel="stylesheet">
    <meta name="robots" content="noindex, nofollow">
</head>
<body>
    <div class="container">
        <div class="row justify-content-center mt-5">
            <div class="col-12 col-md-6 col-lg-4">
                <div class="card shadow">
                    <div class="card-body p-4">
                        <!-- ヘッダー -->
                        <div class="text-center mb-4">
                            <i class="bi bi-person-circle text-primary" style="font-size: 3rem;"></i>
                            <h4 class="mt-3"><%= student.name %> さん</h4>
                            <p class="text-muted">プロフィール編集にはログインが必要です</p>
                        </div>
                        
                        <!-- エラーメッセージ -->
                        <% if (error) { %>
                            <div class="alert alert-danger" role="alert">
                                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                                <%= error %>
                            </div>
                        <% } %>
                        
                        <!-- ログインフォーム -->
                        <form method="POST" action="/student/<%= student.id %>/login">
                            <input type="hidden" name="redirect" value="<%= redirect %>">
                            
                            <div class="mb-4">
                                <label for="password" class="form-label">
                                    <i class="bi bi-key-fill me-1"></i>
                                    パスワード
                                </label>
                                <input type="password" 
                                       class="form-control form-control-lg" 
                                       id="password" 
                                       name="password" 
                                       placeholder="管理者から配布されたパスワード"
                                       required 
                                       autofocus>
                                <div class="form-text">
                                    パスワードは管理者から個別に配布されています
                                </div>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <button type="submit" class="btn btn-primary btn-lg">
                                    <i class="bi bi-box-arrow-in-right me-2"></i>
                                    ログイン
                                </button>
                                <a href="/student/<%= student.id %>" class="btn btn-outline-secondary">
                                    <i class="bi bi-arrow-left me-2"></i>
                                    プロフィールに戻る
                                </a>
                            </div>
                        </form>
                        
                        <!-- ヘルプ -->
                        <div class="mt-4 text-center">
                            <small class="text-muted">
                                <i class="bi bi-info-circle me-1"></i>
                                パスワードがわからない場合は管理者にお問い合わせください
                            </small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
```

### 3. 学生プロフィールページ修正

#### views/student-profile.ejs の編集ボタン部分を修正
```html
<!-- 編集ボタン（条件付き表示） -->
<div class="d-grid gap-2">
    <% if (canEdit) { %>
        <a href="/student/<%= student.id %>/edit" class="btn btn-outline-primary">
            <i class="bi bi-pencil-square me-1"></i>
            プロフィール編集
        </a>
        
        <% if (typeof isAdmin === 'undefined' || !isAdmin) { %>
            <!-- 学生用ログアウトボタン -->
            <a href="/student/<%= student.id %>/logout" class="btn btn-outline-secondary btn-sm">
                <i class="bi bi-box-arrow-right me-1"></i>
                ログアウト
            </a>
        <% } %>
    <% } else { %>
        <!-- 未ログインの場合はログインボタン表示 -->
        <a href="/student/<%= student.id %>/login" class="btn btn-outline-primary">
            <i class="bi bi-key-fill me-1"></i>
            ログインして編集
        </a>
    <% } %>
    
    <a href="/" class="btn btn-outline-secondary btn-sm">
        <i class="bi bi-arrow-left me-1"></i>
        一覧に戻る
    </a>
</div>
```

### 4. セッション設定確認

#### app.js または server.js でセッション設定確認
```javascript
const session = require('express-session');

// セッション設定
app.use(session({
    secret: process.env.SESSION_SECRET || 'your-secret-key-here',
    resave: false,
    saveUninitialized: false,
    cookie: {
        secure: process.env.NODE_ENV === 'production', // 本番環境ではtrue
        maxAge: 24 * 60 * 60 * 1000 // 24時間
    }
}));
```

## ✅ 修正確認チェックリスト

### 認証機能
- [ ] 学生ログインページ作成
- [ ] ログイン処理実装
- [ ] ログアウト機能実装
- [ ] セッション管理実装

### 権限制御
- [ ] 編集ページへのアクセス制限
- [ ] 本人確認ロジック
- [ ] 管理者権限チェック
- [ ] 編集ボタンの条件表示

### テスト項目
- [ ] 未ログイン状態で編集ページアクセス → ログインページへリダイレクト
- [ ] 正しいパスワードでログイン → 編集可能
- [ ] 間違ったパスワード → エラーメッセージ
- [ ] 他の学生のページ編集試行 → アクセス拒否
- [ ] 管理者ログイン時 → 全学生の編集可能

## 📝 完了報告

修正完了時に以下を報告してください：
1. **実装した認証機能の詳細**
2. **権限制御のテスト結果**
3. **既存機能への影響確認**

---

**重要:** この修正により、学生は管理者から配布されたパスワードでログインしないと自分のプロフィールを編集できなくなります。