# Claude Code向け ナビゲーションメニュー追加 指示書

## 🎯 機能概要
サイト上部（ヘッダー直下）に各セクションへジャンプできるナビゲーションメニューを追加

## 📋 実装要件

### メニュー項目（公開サイト）
1. 🔔 お知らせ
2. 📅 行程表
3. 👥 参加者一覧
4. 📋 必要情報

### デザイン要件
- **固定メニュー（sticky）**: スクロールしても上部に固定
- **スムーズスクロール**: クリック時にアニメーション付きで移動
- **レスポンシブ対応**: PC・タブレット・スマホで最適表示
- **アクティブ表示**: 現在表示中のセクションをハイライト

## 🎨 実装詳細

### 1. HTML構造

#### views/index.ejs に追加
```html
<!DOCTYPE html>
<html lang="ja">
<head>
    <!-- 既存のhead内容 -->
</head>
<body>
    <!-- ヘッダー -->
    <header class="site-header">
        <div class="container">
            <h1>2025年度 国際ロータリー第２５９０地区インターアクト研修旅行</h1>
            <p class="subtitle">2025年12月26日（金）～28日（日）</p>
        </div>
    </header>
    
    <!-- ナビゲーションメニュー（新規追加） -->
    <nav class="sticky-nav" id="stickyNav">
        <div class="container">
            <ul class="nav-menu">
                <li class="nav-item">
                    <a href="#news" class="nav-link active">
                        <i class="bi bi-megaphone-fill"></i>
                        <span>お知らせ</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="#schedule" class="nav-link">
                        <i class="bi bi-calendar-event"></i>
                        <span>行程表</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="#participants" class="nav-link">
                        <i class="bi bi-people-fill"></i>
                        <span>参加者</span>
                    </a>
                </li>
                <li class="nav-item">
                    <a href="#information" class="nav-link">
                        <i class="bi bi-info-circle-fill"></i>
                        <span>必要情報</span>
                    </a>
                </li>
            </ul>
        </div>
    </nav>
    
    <!-- メインコンテンツ -->
    <main class="main-content">
        <div class="container">
            <!-- お知らせセクション -->
            <section id="news" class="content-section">
                <h2 class="section-title">
                    <i class="bi bi-megaphone-fill text-primary"></i>
                    お知らせ
                </h2>
                <!-- お知らせ内容 -->
            </section>
            
            <!-- 行程表セクション -->
            <section id="schedule" class="content-section">
                <h2 class="section-title">
                    <i class="bi bi-calendar-event text-success"></i>
                    行程表
                </h2>
                <!-- 行程表内容 -->
            </section>
            
            <!-- 参加者セクション -->
            <section id="participants" class="content-section">
                <h2 class="section-title">
                    <i class="bi bi-people-fill text-info"></i>
                    参加者一覧
                </h2>
                <!-- 参加者内容 -->
            </section>
            
            <!-- 必要情報セクション -->
            <section id="information" class="content-section">
                <h2 class="section-title">
                    <i class="bi bi-info-circle-fill text-warning"></i>
                    必要情報
                </h2>
                <!-- 必要情報内容 -->
            </section>
        </div>
    </main>
    
    <!-- フッター -->
    <footer class="site-footer">
        <!-- フッター内容 -->
    </footer>
    
    <script src="/js/navigation.js"></script>
</body>
</html>
```

### 2. CSS スタイリング

#### public/css/style.css に追加
```css
/* ========================================
   ナビゲーションメニュー
======================================== */

/* Stickyナビゲーション */
.sticky-nav {
    position: sticky;
    top: 0;
    z-index: 1000;
    background: linear-gradient(135deg, #2590ff 0%, #1e7dd8 100%);
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    transition: all 0.3s ease;
}

.sticky-nav.scrolled {
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
}

/* ナビゲーションメニュー */
.nav-menu {
    display: flex;
    justify-content: center;
    align-items: center;
    list-style: none;
    margin: 0;
    padding: 0;
    gap: 0;
}

.nav-item {
    flex: 1;
    max-width: 250px;
}

.nav-link {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 15px 20px;
    color: rgba(255, 255, 255, 0.9);
    text-decoration: none;
    transition: all 0.3s ease;
    border-bottom: 3px solid transparent;
    gap: 5px;
}

.nav-link i {
    font-size: 1.5rem;
    transition: transform 0.3s ease;
}

.nav-link span {
    font-size: 0.9rem;
    font-weight: 500;
}

/* ホバー効果 */
.nav-link:hover {
    background: rgba(255, 255, 255, 0.1);
    color: #ffffff;
}

.nav-link:hover i {
    transform: translateY(-3px);
}

/* アクティブ状態 */
.nav-link.active {
    background: rgba(255, 255, 255, 0.2);
    color: #ffffff;
    border-bottom-color: #ffffff;
    font-weight: 600;
}

/* セクション */
.content-section {
    padding: 60px 0;
    min-height: 400px;
    scroll-margin-top: 80px; /* ナビゲーション高さ分の余白 */
}

.section-title {
    font-size: 2rem;
    margin-bottom: 30px;
    padding-bottom: 15px;
    border-bottom: 3px solid #2590ff;
    display: flex;
    align-items: center;
    gap: 15px;
}

/* レスポンシブ対応 */
@media (max-width: 768px) {
    .nav-link {
        padding: 12px 10px;
        flex-direction: row;
        gap: 8px;
    }
    
    .nav-link i {
        font-size: 1.2rem;
    }
    
    .nav-link span {
        font-size: 0.8rem;
    }
    
    .section-title {
        font-size: 1.5rem;
    }
}

@media (max-width: 576px) {
    .nav-menu {
        gap: 0;
    }
    
    .nav-link {
        padding: 10px 5px;
    }
    
    .nav-link span {
        display: none; /* スマホではアイコンのみ */
    }
    
    .nav-link i {
        font-size: 1.5rem;
    }
}

/* スクロールバーのスタイリング（オプション） */
html {
    scroll-behavior: smooth;
}

::-webkit-scrollbar {
    width: 10px;
}

::-webkit-scrollbar-track {
    background: #f1f1f1;
}

::-webkit-scrollbar-thumb {
    background: #2590ff;
    border-radius: 5px;
}

::-webkit-scrollbar-thumb:hover {
    background: #1e7dd8;
}
```

### 3. JavaScript 機能

#### public/js/navigation.js（新規作成）
```javascript
// ========================================
// ナビゲーション機能
// ========================================

document.addEventListener('DOMContentLoaded', function() {
    initNavigation();
});

function initNavigation() {
    // スムーズスクロール
    setupSmoothScroll();
    
    // スクロール時のアクティブ表示
    setupScrollSpy();
    
    // スクロール時のナビゲーションスタイル変更
    setupStickyNavScroll();
}

// スムーズスクロール機能
function setupSmoothScroll() {
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            const targetId = this.getAttribute('href');
            const targetSection = document.querySelector(targetId);
            
            if (targetSection) {
                const navHeight = document.querySelector('.sticky-nav').offsetHeight;
                const targetPosition = targetSection.offsetTop - navHeight;
                
                window.scrollTo({
                    top: targetPosition,
                    behavior: 'smooth'
                });
                
                // アクティブ状態を即座に更新
                updateActiveLink(targetId);
            }
        });
    });
}

// スクロールスパイ（現在表示中のセクションをハイライト）
function setupScrollSpy() {
    const sections = document.querySelectorAll('.content-section');
    const navLinks = document.querySelectorAll('.nav-link');
    const navHeight = document.querySelector('.sticky-nav').offsetHeight;
    
    window.addEventListener('scroll', () => {
        let currentSection = '';
        
        sections.forEach(section => {
            const sectionTop = section.offsetTop - navHeight - 100;
            const sectionHeight = section.offsetHeight;
            const scrollPosition = window.pageYOffset;
            
            if (scrollPosition >= sectionTop && scrollPosition < sectionTop + sectionHeight) {
                currentSection = '#' + section.getAttribute('id');
            }
        });
        
        if (currentSection) {
            updateActiveLink(currentSection);
        }
    });
}

// アクティブリンク更新
function updateActiveLink(targetId) {
    const navLinks = document.querySelectorAll('.nav-link');
    
    navLinks.forEach(link => {
        link.classList.remove('active');
        if (link.getAttribute('href') === targetId) {
            link.classList.add('active');
        }
    });
}

// スクロール時のナビゲーションスタイル変更
function setupStickyNavScroll() {
    const stickyNav = document.querySelector('.sticky-nav');
    let lastScrollTop = 0;
    
    window.addEventListener('scroll', () => {
        const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
        
        if (scrollTop > 100) {
            stickyNav.classList.add('scrolled');
        } else {
            stickyNav.classList.remove('scrolled');
        }
        
        lastScrollTop = scrollTop;
    });
}

// ページトップボタン（オプション）
function createScrollToTopButton() {
    const button = document.createElement('button');
    button.id = 'scrollToTop';
    button.className = 'scroll-to-top';
    button.innerHTML = '<i class="bi bi-arrow-up-circle-fill"></i>';
    button.style.display = 'none';
    
    document.body.appendChild(button);
    
    window.addEventListener('scroll', () => {
        if (window.pageYOffset > 300) {
            button.style.display = 'block';
        } else {
            button.style.display = 'none';
        }
    });
    
    button.addEventListener('click', () => {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    });
}

// オプション：ページトップボタンを有効化する場合
// createScrollToTopButton();
```

#### ページトップボタンのCSS（オプション）
```css
/* ========================================
   ページトップボタン（オプション）
======================================== */
.scroll-to-top {
    position: fixed;
    bottom: 30px;
    right: 30px;
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: #2590ff;
    color: white;
    border: none;
    cursor: pointer;
    font-size: 1.5rem;
    box-shadow: 0 4px 10px rgba(0, 0, 0, 0.2);
    transition: all 0.3s ease;
    z-index: 999;
}

.scroll-to-top:hover {
    background: #1e7dd8;
    transform: translateY(-5px);
    box-shadow: 0 6px 15px rgba(0, 0, 0, 0.3);
}

@media (max-width: 768px) {
    .scroll-to-top {
        width: 45px;
        height: 45px;
        bottom: 20px;
        right: 20px;
    }
}
```

## ✅ 実装チェックリスト

### HTML
- [ ] 各セクションに id 属性追加
- [ ] ナビゲーションメニュー追加
- [ ] Bootstrap Icons 読み込み確認

### CSS
- [ ] sticky-nav スタイル実装
- [ ] レスポンシブ対応
- [ ] ホバー・アクティブ効果

### JavaScript
- [ ] スムーズスクロール動作
- [ ] スクロールスパイ動作
- [ ] アクティブリンク更新

## 🧪 テスト項目

### 基本機能
- [ ] 各メニューリンククリックで対応セクションに移動
- [ ] スクロール時に現在のセクションがハイライト
- [ ] ナビゲーションが上部に固定される

### レスポンシブ
- [ ] PC表示：アイコン+テキスト
- [ ] タブレット表示：適切なサイズ調整
- [ ] スマホ表示：アイコンのみ

### UX
- [ ] スムーズなスクロールアニメーション
- [ ] 視覚的なフィードバック（ホバー・アクティブ）
- [ ] 適切なスクロール位置（ナビゲーション高さ考慮）

## 📝 完了報告

実装完了時に以下を報告してください：
1. **実装した機能の詳細**
2. **各デバイスでの動作確認結果**
3. **UX改善の追加提案（あれば）**

---

**オプション機能:**
- ページトップボタン（スクロール時に表示）
- ハンバーガーメニュー（スマホ用）
- サブメニュー展開（参加者カテゴリ別など）