# Claude Code向け ダミーデータ自動生成停止 指示書

## 🎯 問題点
サーバー再起動やコード更新のたびにダミーデータが自動生成されてしまう

## 🔧 修正方針
データベース初期化処理を修正し、既存データがある場合はダミーデータを生成しないようにする

## 📋 修正手順

### 1. データベース初期化スクリプトの確認

#### scripts/init-database.js の修正
```javascript
const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcrypt');
const db = new sqlite3.Database('./database.sqlite');

async function initDatabase() {
    console.log('データベース初期化を開始...');
    
    // テーブル作成（CREATE TABLE IF NOT EXISTS で既存保持）
    await createTables();
    
    // 既存データ確認
    const existingDataCheck = await checkExistingData();
    
    if (existingDataCheck.hasData) {
        console.log('既存データが見つかりました。ダミーデータの生成をスキップします。');
        console.log(`- お知らせ: ${existingDataCheck.newsCount}件`);
        console.log(`- 行程表: ${existingDataCheck.schedulesCount}件`);
        console.log(`- 参加者: ${existingDataCheck.participantsCount}件`);
        console.log(`- 必要情報: ${existingDataCheck.informationCount}件`);
        
        // 管理者アカウントのみ確認・作成
        await ensureAdminExists();
        
    } else {
        console.log('既存データなし。初期データを作成します...');
        
        // 管理者アカウント作成
        await createAdminAccount();
        
        // サンプルデータ作成（初回のみ）
        await createSampleData();
        
        console.log('初期データの作成が完了しました。');
    }
    
    console.log('データベース初期化完了！');
}

// テーブル作成関数
async function createTables() {
    return new Promise((resolve, reject) => {
        db.serialize(() => {
            // お知らせテーブル
            db.run(`CREATE TABLE IF NOT EXISTS news (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                title TEXT NOT NULL,
                content TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                is_published BOOLEAN DEFAULT 1
            )`);
            
            // 行程表テーブル
            db.run(`CREATE TABLE IF NOT EXISTS schedules (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date DATE NOT NULL,
                time TIME,
                title TEXT NOT NULL,
                description TEXT,
                location TEXT,
                order_index INTEGER DEFAULT 0
            )`);
            
            // 参加者テーブル
            db.run(`CREATE TABLE IF NOT EXISTS participants (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                type TEXT CHECK(type IN ('student', 'companion', 'chaperone')) NOT NULL,
                school_name TEXT,
                grade TEXT,
                group_name TEXT CHECK(group_name IN ('A', 'B', 'C', 'D', 'E', 'F')),
                photo_path TEXT,
                password TEXT,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )`);
            
            // 学生プロフィールテーブル
            db.run(`CREATE TABLE IF NOT EXISTS student_profiles (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                participant_id INTEGER UNIQUE,
                participation_comment TEXT,
                hobby_selfpr TEXT,
                prestudy_task TEXT,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (participant_id) REFERENCES participants(id)
            )`);
            
            // 必要情報テーブル
            db.run(`CREATE TABLE IF NOT EXISTS information (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                category TEXT NOT NULL,
                title TEXT NOT NULL,
                content TEXT NOT NULL,
                order_index INTEGER DEFAULT 0
            )`);
            
            // 管理者テーブル
            db.run(`CREATE TABLE IF NOT EXISTS admins (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT UNIQUE NOT NULL,
                password_hash TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )`, (err) => {
                if (err) reject(err);
                else resolve();
            });
        });
    });
}

// 既存データ確認関数
async function checkExistingData() {
    return new Promise((resolve, reject) => {
        const counts = {};
        
        db.get('SELECT COUNT(*) as count FROM news', (err, row) => {
            if (err) return reject(err);
            counts.newsCount = row.count;
            
            db.get('SELECT COUNT(*) as count FROM schedules', (err, row) => {
                if (err) return reject(err);
                counts.schedulesCount = row.count;
                
                db.get('SELECT COUNT(*) as count FROM participants', (err, row) => {
                    if (err) return reject(err);
                    counts.participantsCount = row.count;
                    
                    db.get('SELECT COUNT(*) as count FROM information', (err, row) => {
                        if (err) return reject(err);
                        counts.informationCount = row.count;
                        
                        // いずれかのテーブルにデータがあればtrue
                        counts.hasData = counts.newsCount > 0 || 
                                        counts.schedulesCount > 0 || 
                                        counts.participantsCount > 0 || 
                                        counts.informationCount > 0;
                        
                        resolve(counts);
                    });
                });
            });
        });
    });
}

// 管理者アカウント確認・作成
async function ensureAdminExists() {
    return new Promise((resolve, reject) => {
        db.get('SELECT * FROM admins WHERE username = ?', ['admin'], async (err, row) => {
            if (err) return reject(err);
            
            if (!row) {
                console.log('管理者アカウントが存在しません。作成します...');
                await createAdminAccount();
            } else {
                console.log('管理者アカウントは既に存在します。');
            }
            resolve();
        });
    });
}

// 管理者アカウント作成
async function createAdminAccount() {
    return new Promise(async (resolve, reject) => {
        const passwordHash = await bcrypt.hash('dev2590', 10);
        
        db.run(
            'INSERT OR IGNORE INTO admins (username, password_hash) VALUES (?, ?)',
            ['admin', passwordHash],
            (err) => {
                if (err) reject(err);
                else {
                    console.log('管理者アカウント作成: admin / dev2590');
                    resolve();
                }
            }
        );
    });
}

// サンプルデータ作成（初回のみ）
async function createSampleData() {
    return new Promise((resolve, reject) => {
        db.serialize(() => {
            // サンプルお知らせ
            db.run(`INSERT INTO news (title, content) VALUES 
                ('説明会開催のお知らせ', '12月6日に研修旅行の説明会を開催します。'),
                ('事前課題について', '事前課題の提出期限は11月30日です。'),
                ('持ち物リスト更新', '必要情報ページを確認してください。')
            `);
            
            // サンプル行程表
            db.run(`INSERT INTO schedules (date, time, title, description, location, order_index) VALUES
                ('2025-12-26', '09:00', '集合', '研修旅行スタート', '東京駅', 1),
                ('2025-12-26', '12:00', '昼食', 'ランチタイム', 'レストラン', 2),
                ('2025-12-27', '10:00', '研修プログラム', 'ワークショップ', '研修施設', 3),
                ('2025-12-28', '15:00', '解散', '研修旅行終了', '東京駅', 4)
            `);
            
            // サンプル参加者（学生3名、同行者1名、引率者1名）
            db.run(`INSERT INTO participants (name, type, school_name, grade, group_name) VALUES
                ('山田太郎', 'student', '東京高校', '2年', 'A'),
                ('佐藤花子', 'student', '神奈川高校', '3年', 'B'),
                ('鈴木一郎', 'student', '千葉高校', '2年', 'A'),
                ('田中次郎', 'companion', NULL, NULL, NULL),
                ('高橋先生', 'chaperone', NULL, NULL, NULL)
            `);
            
            // サンプル必要情報
            db.run(`INSERT INTO information (category, title, content, order_index) VALUES
                ('持ち物', '基本的な持ち物', '着替え、洗面用具、筆記用具', 1),
                ('注意事項', '集合時間厳守', '遅刻しないようお願いします', 2),
                ('緊急連絡先', '緊急時の連絡先', '担当者: 03-1234-5678', 3)
            `, (err) => {
                if (err) reject(err);
                else {
                    console.log('サンプルデータ作成完了');
                    resolve();
                }
            });
        });
    });
}

// 実行
initDatabase()
    .then(() => {
        db.close();
        console.log('データベース接続を閉じました');
    })
    .catch((error) => {
        console.error('エラー:', error);
        db.close();
    });
```

### 2. サーバー起動時の自動初期化を確認

#### app.js または server.js の確認
```javascript
// サーバー起動時にデータベース初期化を実行している箇所を探す

// 修正前（毎回実行される可能性）
const initDb = require('./scripts/init-database');
initDb(); // ← これが毎回実行されていると問題

// 修正後（条件付き実行）
const fs = require('fs');
const path = require('path');

// データベースファイルが存在しない場合のみ初期化
const dbPath = path.join(__dirname, 'database.sqlite');
if (!fs.existsSync(dbPath)) {
    console.log('データベースファイルが存在しません。初期化します...');
    const initDb = require('./scripts/init-database');
    initDb();
} else {
    console.log('既存のデータベースを使用します。');
}
```

### 3. package.jsonスクリプトの整理

```json
{
  "scripts": {
    "start": "node app.js",
    "dev": "nodemon app.js",
    "init-db": "node scripts/init-database.js",
    "reset-db": "rm -f database.sqlite && node scripts/init-database.js",
    "backup-db": "cp database.sqlite database.backup.$(date +%Y%m%d_%H%M%S).sqlite"
  }
}
```

**使い方：**
- `npm run dev` - 通常の開発（データ保持）
- `npm run init-db` - データベース初期化のみ（既存データチェックあり）
- `npm run reset-db` - 完全リセット（データ削除 + 再作成）
- `npm run backup-db` - データベースバックアップ

### 4. 開発時の運用ルール

#### データ保持の確認
```bash
# 開発開始時にデータベース確認
sqlite3 database.sqlite "SELECT COUNT(*) FROM participants;"

# バックアップ作成（念のため）
npm run backup-db

# 通常の開発サーバー起動（データ保持）
npm run dev
```

#### 完全リセットが必要な場合のみ
```bash
# データを完全に削除して再作成したい場合のみ実行
npm run reset-db
```

## ✅ 修正確認チェックリスト

### データ保持確認
- [ ] サーバー再起動後もデータが残っている
- [ ] コード変更後もデータが保持される
- [ ] `npm run dev` でダミーデータが増えない

### 初期化機能確認
- [ ] `npm run init-db` で既存データチェックされる
- [ ] `npm run reset-db` で完全リセットできる
- [ ] 管理者アカウントは常に存在する

### 運用確認
- [ ] 新規環境では自動的にサンプルデータ作成
- [ ] 既存環境ではデータ保持
- [ ] バックアップ作成が簡単にできる

## 📝 完了報告

修正完了時に以下を報告してください：
1. **修正したファイル一覧**
2. **データ保持の動作確認結果**
3. **init-db, reset-db スクリプトの動作確認**

---

**重要:** 既存のデータベースを保護するため、修正前に必ずバックアップを取得してください。
```bash
cp database.sqlite database.backup.sqlite
```