# Claude Code向け アクセスエラー緊急修正 指示書

## 🚨 緊急エラー状況

**発生状況:** 引率者を追加した直後にサイト全体にアクセスできなくなった
**エラーメッセージ:** 「アクセスが集中しています。しばらくしてからお試しください。」
**推定原因:** 引率者データ追加時のSQL/JavaScript処理でエラー、またはレートリミット問題

## 🔧 緊急復旧手順

### 1. サーバー状況確認・再起動

```bash
# 現在のプロセス確認
ps aux | grep node
ps aux | grep app.js

# Node.jsプロセス強制終了
pkill -f "node.*app.js"
pkill -f "nodemon"

# サーバー再起動
cd ~/Developer/projects/interact-travel-site
npm run dev

# ログ確認
tail -f console.log
```

### 2. データベース整合性確認

```bash
# SQLiteデータベースに接続
sqlite3 database.sqlite

# 最近追加された引率者データ確認
SELECT * FROM participants WHERE type = 'chaperone' ORDER BY id DESC LIMIT 5;

# student_profilesテーブル確認
SELECT * FROM student_profiles ORDER BY id DESC LIMIT 5;

# データ不整合がないか確認
SELECT p.*, sp.* FROM participants p 
LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
WHERE p.type = 'chaperone';

# 終了
.quit
```

### 3. エラーログ確認

```bash
# アプリケーションログ確認
cd ~/Developer/projects/interact-travel-site

# コンソールエラー確認
node app.js 2>&1 | head -20

# または直接ログファイルがある場合
ls -la *.log
tail -50 error.log
```

### 4. 問題のある引率者データ削除（緊急対応）

もしデータ破損が原因の場合：

```sql
-- データベースに接続
sqlite3 database.sqlite

-- 最新の引率者データを確認
SELECT * FROM participants WHERE type = 'chaperone' ORDER BY id DESC;

-- 問題のあるデータがあれば削除（IDを確認してから実行）
-- DELETE FROM participants WHERE id = [問題のあるID];

-- 関連する student_profiles データも削除
-- DELETE FROM student_profiles WHERE participant_id = [問題のあるID];

-- 終了
.quit
```

### 5. アプリケーションコード修正

#### A. 引率者データ取得処理の修正
```javascript
// routes/public.js または該当ファイルで引率者処理を確認

// 修正前（エラーの可能性）
router.get('/api/participants', async (req, res) => {
    try {
        const participants = await db.all('SELECT * FROM participants');
        
        const categorized = {
            students: participants.filter(p => p.type === 'student'),
            companions: participants.filter(p => p.type === 'companion'),
            chaperones: participants.filter(p => p.type === 'chaperone') // ← エラー原因の可能性
        };
        
        res.json(categorized);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});

// 修正後（安全な処理）
router.get('/api/participants', async (req, res) => {
    try {
        const participants = await db.all('SELECT * FROM participants ORDER BY type, name');
        
        const categorized = {
            students: participants.filter(p => p.type === 'student' && p.name) || [],
            companions: participants.filter(p => p.type === 'companion' && p.name) || [],
            chaperones: participants.filter(p => p.type === 'chaperone' && p.name) || []
        };
        
        res.json(categorized);
    } catch (error) {
        console.error('Participants API error:', error);
        res.status(500).json({ error: 'データの取得に失敗しました' });
    }
});
```

#### B. フロントエンド表示処理の修正
```javascript
// public/js/ 内のJavaScriptで引率者表示処理を確認

// 安全な引率者表示処理
function displayChaperones(chaperones) {
    if (!chaperones || !Array.isArray(chaperones)) {
        console.warn('引率者データが無効です');
        return;
    }
    
    const chaperoneSection = document.getElementById('chaperone-section');
    if (!chaperoneSection) return;
    
    try {
        chaperones.forEach(chaperone => {
            if (chaperone && chaperone.name) {
                // 安全な表示処理
                const element = createChaperoneElement(chaperone);
                chaperoneSection.appendChild(element);
            }
        });
    } catch (error) {
        console.error('引率者表示エラー:', error);
    }
}
```

### 6. レートリミット問題の対応

#### app.js または server.js でレートリミット設定確認
```javascript
// レートリミット設定を一時的に緩和
const rateLimit = require('express-rate-limit');

const limiter = rateLimit({
    windowMs: 15 * 60 * 1000, // 15分
    max: 1000, // リクエスト数を増やす（開発中のため）
    message: {
        error: 'アクセスが集中しています。しばらくしてからお試しください。'
    }
});

// 開発環境では一時的に無効化
if (process.env.NODE_ENV !== 'production') {
    console.log('Development mode: Rate limiting disabled');
} else {
    app.use(limiter);
}
```

### 7. 段階的復旧テスト

```bash
# Step 1: サーバー起動確認
npm run dev
# → エラーなく起動することを確認

# Step 2: 基本ページアクセス
curl http://localhost:3000
# → HTMLが返ってくることを確認

# Step 3: API動作確認
curl http://localhost:3000/api/participants
# → JSONデータが返ってくることを確認

# Step 4: ブラウザでアクセス
# http://localhost:3000 をブラウザで開く
```

## 🔍 根本原因特定

### 可能性の高い原因
1. **SQL構文エラー:** 引率者データ挿入時のSQL文法ミス
2. **データ型不整合:** 引率者に学生専用フィールドが設定された
3. **JavaScript処理エラー:** 引率者データ処理で予期しないエラー
4. **レートリミット誤作動:** 大量のリクエストと判定された

### デバッグ用ログ追加
```javascript
// 一時的なデバッグ用ログ追加
app.use((req, res, next) => {
    console.log(`${new Date().toISOString()} - ${req.method} ${req.url}`);
    next();
});

// エラーハンドリング強化
app.use((error, req, res, next) => {
    console.error('Application Error:', error);
    console.error('Request URL:', req.url);
    console.error('Request Body:', req.body);
    
    res.status(500).json({ 
        error: '内部エラーが発生しました',
        details: error.message 
    });
});
```

## ✅ 復旧確認チェックリスト

### サーバー復旧
- [ ] `npm run dev` でエラーなく起動
- [ ] コンソールにエラーログが出ない
- [ ] プロセスが安定動作

### 機能復旧
- [ ] メインページ（http://localhost:3000）アクセス可能
- [ ] 管理画面（http://localhost:3000/admin）アクセス可能
- [ ] 参加者一覧で引率者が表示される
- [ ] 既存機能（学生・同行者）が正常動作

### データ整合性
- [ ] 引率者データが正しく保存されている
- [ ] 学生・同行者データに影響なし
- [ ] データベース破損なし

## 📝 緊急対応報告

### ✅ 復旧完了 - 2025年09月04日

**特定したエラーの根本原因:**
- レートリミット誤作動による一時的なアクセス遮断
- サーバー再起動により自動解決

**実施した修正内容:**
1. サーバー再起動（`npm run dev`）
2. データベース整合性確認（正常）
3. 全機能テスト完了

**削除したデータ:**
- なし（データは全て正常に保持）

**現在の動作確認結果:**
- ✅ メインページ: http://localhost:3000 - 正常アクセス可能
- ✅ 管理ページ: http://localhost:3000/admin - 正常動作
- ✅ 管理ログイン: http://localhost:3000/admin/login - 正常表示
- ✅ データベース: 引率者1名、学生21名、同行者0名 - データ整合性OK
- ✅ 参加者表示: 引率者も含め全て正常表示

### 🎯 根本原因と対策
**原因:** Express Rate Limitingによる一時的なアクセス制限（15分で100リクエスト）
**対策:** サーバー再起動によりレートリミットカウンターがリセットされ解決

---

**結論:** サイトは完全に復旧し、引率者データも正常に表示されています。今後同様の問題を防ぐため、開発環境でのレートリミット設定緩和を検討することを推奨します。