# Claude Code向け SQLiteエラー緊急修正 指示書

## 🚨 緊急エラー

**エラー内容:**
```
SQLITE_ERROR: table student_profiles has no column named hobbies
```

**原因:** アプリケーションコードが存在しない `hobbies` カラムを参照している

## 🔧 緊急修正手順

### 1. エラー箇所特定
以下のコマンドでエラー箇所を特定してください：

```bash
# プロジェクト全体で 'hobbies' を検索
grep -r "hobbies" --include="*.js" --include="*.ejs" .

# 'hobby' も確認
grep -r "hobby" --include="*.js" --include="*.ejs" .

# 'motivation' も確認  
grep -r "motivation" --include="*.js" --include="*.ejs" .
```

### 2. 現在のデータベース構造確認

```bash
# SQLiteデータベースに接続
sqlite3 database.sqlite

# テーブル構造確認
.schema student_profiles

# 現在のデータ確認
SELECT * FROM student_profiles LIMIT 3;

# 終了
.quit
```

### 3. 修正対象ファイル（推測）

#### A. app.js または server.js
```javascript
// 古いフィールド参照を削除・修正
// 例：
// 修正前（エラーの原因）
const student = await db.get(`
    SELECT p.*, sp.hobbies, sp.motivation, sp.participation_comment
    FROM participants p 
    LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
    WHERE p.id = ?
`, [id]);

// 修正後
const student = await db.get(`
    SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
    FROM participants p 
    LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
    WHERE p.id = ?
`, [id]);
```

#### B. routes/student.js
```javascript
// 古いフィールド参照を新フィールドに置換

// 削除対象の例
router.get('/:id', async (req, res) => {
    const student = await db.get(`
        SELECT p.*, sp.hobbies, sp.motivation  // ← これが原因
        FROM participants p 
        LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
        WHERE p.id = ? AND p.type = 'student'
    `, [studentId]);
});

// 修正後
router.get('/:id', async (req, res) => {
    const student = await db.get(`
        SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
        FROM participants p 
        LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
        WHERE p.id = ? AND p.type = 'student'
    `, [studentId]);
});
```

#### C. routes/admin.js
```javascript
// 管理画面での古いフィールド参照を削除

// 学生一覧取得
router.get('/api/students', async (req, res) => {
    // 修正前
    const students = await db.all(`
        SELECT p.*, sp.hobbies, sp.motivation  // ← エラー原因
        FROM participants p 
        LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
        WHERE p.type = 'student'
    `);
    
    // 修正後
    const students = await db.all(`
        SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
        FROM participants p 
        LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
        WHERE p.type = 'student'
    `);
});
```

### 4. データベース初期化スクリプト修正

#### scripts/init-database.js
```javascript
// student_profilesテーブル作成部分を確認・修正

// 正しいテーブル作成SQL
const createStudentProfilesTable = `
CREATE TABLE IF NOT EXISTS student_profiles (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    participant_id INTEGER UNIQUE,
    participation_comment TEXT,
    hobby_selfpr TEXT,
    prestudy_task TEXT,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (participant_id) REFERENCES participants(id)
)`;

// 古いフィールド（hobby, motivation, hobbies）を参照するコードを削除
```

### 5. 一時的回避策（緊急対応）

もしすぐに修正箇所が特定できない場合は、一時的に古いカラムを追加：

```sql
-- データベースに接続
sqlite3 database.sqlite

-- 一時的に古いカラムを追加（エラー回避）
ALTER TABLE student_profiles ADD COLUMN hobbies TEXT DEFAULT '';
ALTER TABLE student_profiles ADD COLUMN hobby TEXT DEFAULT '';
ALTER TABLE student_profiles ADD COLUMN motivation TEXT DEFAULT '';

-- 確認
.schema student_profiles

-- 終了
.quit
```

**⚠️ 注意:** これは一時的な回避策です。根本的には古いフィールドを参照するコードを修正する必要があります。

### 6. 修正完了後の確認

```bash
# サーバー再起動
npm run dev

# エラーが解消されることを確認
# ブラウザで http://localhost:3000 アクセス
# ログにエラーが出ないことを確認
```

## 🔍 デバッグ手順

### Step 1: 詳細エラーログ確認
```javascript
// app.js または server.js にデバッグ用のエラーハンドリング追加
process.on('uncaughtException', (error) => {
    console.error('Uncaught Exception:', error);
    console.error('Stack trace:', error.stack);
});

process.on('unhandledRejection', (reason, promise) => {
    console.error('Unhandled Rejection at:', promise, 'reason:', reason);
});
```

### Step 2: SQL文デバッグ
```javascript
// データベースクエリ実行前にログ出力
const loggedDb = {
    get: async (sql, params) => {
        console.log('SQL:', sql);
        console.log('Params:', params);
        return await db.get(sql, params);
    },
    all: async (sql, params) => {
        console.log('SQL:', sql);
        console.log('Params:', params);
        return await db.all(sql, params);
    }
};
```

## ✅ 修正チェックリスト

### コード修正
- [ ] 全ての `hobbies` 参照を削除・修正
- [ ] 全ての `hobby` 参照を削除・修正
- [ ] 全ての `motivation` 参照を削除・修正
- [ ] 正しいフィールド名（`participation_comment`, `hobby_selfpr`, `prestudy_task`）に統一

### データベース確認
- [ ] `student_profiles` テーブル構造が正しい
- [ ] 不要なカラムが存在しない
- [ ] 必要なカラムがすべて存在する

### 動作確認
- [ ] `npm run dev` でエラーなく起動
- [ ] ブラウザでサイトアクセス可能
- [ ] 学生個別ページ表示可能
- [ ] 学生編集機能動作

## 📝 修正報告

修正完了時に以下を報告してください：
1. **発見・修正したファイル一覧**
2. **修正したSQL文**
3. **最終的なテーブル構造**
4. **動作確認結果**

---

**最優先:** まずは `grep` コマンドで `hobbies` を検索し、参照箇所をすべて特定・修正してください。これでエラーは解消されるはずです。