# Claude Code向け 既存フィールド削除 指示書

## 🎯 削除対象
既存の `student_profiles` テーブルの以下フィールドを削除：
- **hobby** - 趣味フィールド
- **motivation** - やる気・動機フィールド

**理由:** 新しい3項目（研修参加・趣味自己PR・事前学習）に統合されたため、古いフィールドは不要

## 🔧 削除手順

### 1. データベーステーブル確認
```sql
-- 現在のテーブル構造確認
.schema student_profiles

-- 既存データ確認（削除前バックアップ用）
SELECT * FROM student_profiles LIMIT 5;
```

### 2. データベースからカラム削除

#### SQLiteでのカラム削除（手順）
```sql
-- Step 1: 既存データのバックアップ
CREATE TABLE student_profiles_backup AS SELECT * FROM student_profiles;

-- Step 2: 新しいテーブル構造で再作成
CREATE TABLE student_profiles_new (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    participant_id INTEGER UNIQUE,
    participation_comment TEXT,      -- 研修参加について
    hobby_selfpr TEXT,              -- 趣味・自己PR  
    prestudy_task TEXT,             -- 事前学習課題
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (participant_id) REFERENCES participants(id)
);

-- Step 3: データ移行（既存の有用データがあれば）
INSERT INTO student_profiles_new (
    id, participant_id, participation_comment, hobby_selfpr, prestudy_task, updated_at
)
SELECT 
    id, 
    participant_id,
    -- 既存のmotivationをparticipation_commentに移行（データがある場合）
    CASE 
        WHEN motivation IS NOT NULL AND motivation != '' 
        THEN motivation 
        ELSE participation_comment 
    END as participation_comment,
    -- 既存のhobbyをhobby_selfprに移行（データがある場合）
    CASE 
        WHEN hobby IS NOT NULL AND hobby != '' 
        THEN hobby 
        ELSE hobby_selfpr 
    END as hobby_selfpr,
    prestudy_task,
    updated_at
FROM student_profiles;

-- Step 4: テーブル置換
DROP TABLE student_profiles;
ALTER TABLE student_profiles_new RENAME TO student_profiles;

-- Step 5: 確認
SELECT * FROM student_profiles;
```

### 3. アプリケーションコード確認・削除

#### A. データベースモデル（models/database.js等）
```javascript
// 以下の古いフィールドを参照している箇所をすべて削除
// - hobby
// - motivation

// 例：SQLクエリの修正
// 修正前
const student = await db.get(`
    SELECT p.*, sp.hobby, sp.motivation, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
    FROM participants p 
    LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
    WHERE p.id = ? AND p.type = 'student'
`, [studentId]);

// 修正後
const student = await db.get(`
    SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
    FROM participants p 
    LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
    WHERE p.id = ? AND p.type = 'student'
`, [studentId]);
```

#### B. ルーター（routes/student.js等）
```javascript
// 古いフィールドを含むINSERT/UPDATEクエリを修正

// 修正前（削除対象）
router.post('/:id/update', async (req, res) => {
    const { hobby, motivation, participation_comment, hobby_selfpr, prestudy_task } = req.body;
    
    await db.run(`
        UPDATE student_profiles 
        SET hobby = ?, motivation = ?, participation_comment = ?, hobby_selfpr = ?, prestudy_task = ?
        WHERE participant_id = ?
    `, [hobby, motivation, participation_comment, hobby_selfpr, prestudy_task, studentId]);
});

// 修正後（新フィールドのみ）
router.post('/:id/update', async (req, res) => {
    const { participation_comment, hobby_selfpr, prestudy_task } = req.body;
    
    await db.run(`
        UPDATE student_profiles 
        SET participation_comment = ?, hobby_selfpr = ?, prestudy_task = ?, updated_at = CURRENT_TIMESTAMP
        WHERE participant_id = ?
    `, [participation_comment, hobby_selfpr, prestudy_task, studentId]);
});
```

#### C. 管理画面（views/admin.ejs等）
```html
<!-- 管理画面で古いフィールドを表示・編集している箇所を削除 -->

<!-- 削除対象の例 -->
<div class="form-group">
    <label for="hobby">趣味</label>
    <input type="text" class="form-control" id="hobby" name="hobby" value="<%- student.hobby || '' %>">
</div>

<div class="form-group">
    <label for="motivation">やる気・動機</label>
    <textarea class="form-control" id="motivation" name="motivation"><%- student.motivation || '' %></textarea>
</div>

<!-- 上記のような古いフィールドをすべて削除 -->
```

#### D. 学生編集ページ（views/student-edit.ejs）
```html
<!-- 既に新しい3項目で構成されているはずだが、古いフィールドがあれば削除 -->
<!-- 削除対象：hobby, motivationフィールド -->
```

### 4. データ移行スクリプト（オプション）

#### 既存データを新フィールドに統合
```javascript
// scripts/migrate-student-data.js
const sqlite3 = require('sqlite3').verbose();
const db = new sqlite3.Database('./database.sqlite');

async function migrateStudentData() {
    try {
        console.log('学生データの移行を開始...');
        
        // 既存データで新フィールドが空、古いフィールドにデータがある場合のみ移行
        await db.run(`
            UPDATE student_profiles 
            SET participation_comment = CASE 
                WHEN (participation_comment IS NULL OR participation_comment = '') 
                     AND (motivation IS NOT NULL AND motivation != '') 
                THEN motivation 
                ELSE participation_comment 
            END,
            hobby_selfpr = CASE 
                WHEN (hobby_selfpr IS NULL OR hobby_selfpr = '') 
                     AND (hobby IS NOT NULL AND hobby != '') 
                THEN hobby 
                ELSE hobby_selfpr 
            END
            WHERE (motivation IS NOT NULL AND motivation != '') 
               OR (hobby IS NOT NULL AND hobby != '')
        `);
        
        console.log('データ移行完了');
        
        // 移行後のデータ確認
        const updatedProfiles = await db.all('SELECT * FROM student_profiles');
        console.log('移行後のプロフィール:', updatedProfiles);
        
    } catch (error) {
        console.error('移行エラー:', error);
    } finally {
        db.close();
    }
}

migrateStudentData();
```

### 5. 削除確認・テスト

#### A. データベース構造確認
```sql
-- テーブル構造が正しいか確認
.schema student_profiles

-- 期待される構造：
-- CREATE TABLE student_profiles (
--     id INTEGER PRIMARY KEY AUTOINCREMENT,
--     participant_id INTEGER UNIQUE,
--     participation_comment TEXT,
--     hobby_selfpr TEXT,
--     prestudy_task TEXT,
--     updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
--     FOREIGN KEY (participant_id) REFERENCES participants(id)
-- );
```

#### B. アプリケーション動作確認
```bash
# 開発サーバー起動
npm run dev

# 確認項目：
# 1. 管理画面で学生管理が正常動作
# 2. 学生個別ページ表示が正常
# 3. 学生編集機能が正常動作
# 4. エラーログに古いフィールド参照エラーがない
```

### 6. コード全体検索・削除

#### 削除対象のキーワード検索
```bash
# プロジェクト全体で古いフィールド名を検索
grep -r "hobby" --include="*.js" --include="*.ejs" .
grep -r "motivation" --include="*.js" --include="*.ejs" .

# 見つかった箇所をすべて確認・削除
```

## ✅ 削除完了チェックリスト

### データベース
- [ ] hobby カラム削除完了
- [ ] motivation カラム削除完了  
- [ ] 既存データの移行確認
- [ ] 新テーブル構造で正常動作

### アプリケーションコード
- [ ] SQLクエリから古いフィールド削除
- [ ] ルーター処理の古いフィールド削除
- [ ] テンプレートの古いフィールド削除
- [ ] JavaScript の古いフィールド削除

### 動作確認
- [ ] 管理画面で学生管理正常動作
- [ ] 学生個別ページ表示正常
- [ ] 学生編集機能正常動作
- [ ] エラーログにフィールド関連エラーなし

## 🧪 テスト項目

### 基本機能テスト
1. **学生一覧表示** - エラーなし
2. **学生個別ページ** - 3項目が正しく表示
3. **学生編集機能** - 3項目の編集・保存が正常動作
4. **管理画面** - 学生管理が正常動作

### データ整合性テスト  
1. **既存学生データ** - 削除前後で重要データが保持
2. **新規学生追加** - 問題なく動作
3. **バックアップデータ** - 必要時に復元可能

## 📝 完了報告

削除完了時に以下を報告してください：
1. **削除したフィールドとファイル一覧**
2. **データ移行の結果**
3. **動作確認テスト結果**  
4. **エラーログの確認結果**

## 🎨 追加要望：編集画面デザイン改善

### 問題点
- 入力フィールドが小さすぎてPC作業で使いづらい
- 全体的なデザインが改善の余地あり

### 改善要求
1. **PC用レスポンシブ対応**
   - PC表示時はテキストエリアを大きく
   - スマホ表示時は適切なサイズ維持

2. **UX向上**
   - 入力しやすいレイアウト
   - 視覚的に分かりやすい構成
   - 快適な編集体験

### 具体的改善案

#### A. PC用テキストエリアサイズ調整
```css
/* レスポンシブ対応のテキストエリア */
.student-edit-textarea {
    min-height: 120px;
    font-size: 14px;
    line-height: 1.5;
}

/* PC表示時（768px以上）はより大きく */
@media (min-width: 768px) {
    .student-edit-textarea {
        min-height: 150px;
        font-size: 16px;
    }
}

/* デスクトップ表示時（1200px以上）はさらに大きく */
@media (min-width: 1200px) {
    .student-edit-textarea {
        min-height: 180px;
    }
}
```

#### B. 編集フォームレイアウト改善
```html
<!-- 改善されたstudent-edit.ejsレイアウト例 -->
<div class="container-lg mt-4">
    <div class="row justify-content-center">
        <div class="col-12 col-xl-10">
            <div class="row">
                <!-- サイドバー：学生基本情報 -->
                <div class="col-12 col-lg-4 mb-4">
                    <div class="sticky-top" style="top: 20px;">
                        <!-- 基本情報カード -->
                    </div>
                </div>
                
                <!-- メインコンテンツ：編集フォーム -->
                <div class="col-12 col-lg-8">
                    <div class="card shadow-sm">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">
                                <i class="bi bi-pencil-square me-2"></i>
                                プロフィール編集
                            </h5>
                        </div>
                        <div class="card-body p-4">
                            <!-- 改善されたフォーム -->
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
```

#### C. インタラクティブUI改善
```javascript
// 文字数カウンター・入力支援機能強化
function enhancedTextAreaSetup(textareaId, counterId, maxLength = 1000) {
    const textarea = document.getElementById(textareaId);
    const counter = document.getElementById(counterId);
    
    // 自動リサイズ機能
    function autoResize() {
        textarea.style.height = 'auto';
        textarea.style.height = Math.max(textarea.scrollHeight, 120) + 'px';
    }
    
    // 文字数カウンター
    function updateCounter() {
        const count = textarea.value.length;
        counter.textContent = count;
        
        // 色分け表示
        if (count > maxLength * 0.9) {
            counter.parentElement.classList.add('text-warning');
        }
        if (count > maxLength) {
            counter.parentElement.classList.add('text-danger');
        } else {
            counter.parentElement.classList.remove('text-danger', 'text-warning');
        }
    }
    
    // イベントリスナー
    textarea.addEventListener('input', function() {
        autoResize();
        updateCounter();
    });
    
    // 初期化
    autoResize();
    updateCounter();
}
```

#### D. 視覚的改善要素
```css
/* カード・セクションデザイン改善 */
.edit-section {
    background: linear-gradient(135deg, #f8f9ff 0%, #ffffff 100%);
    border: 1px solid #e3f2fd;
    border-radius: 12px;
    margin-bottom: 2rem;
    transition: all 0.3s ease;
}

.edit-section:hover {
    box-shadow: 0 4px 12px rgba(37, 144, 255, 0.15);
    transform: translateY(-2px);
}

.section-icon {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.2em;
    margin-bottom: 1rem;
}

/* テキストエリア改善 */
.enhanced-textarea {
    border: 2px solid #e9ecef;
    border-radius: 8px;
    transition: all 0.3s ease;
    background: #fafafa;
}

.enhanced-textarea:focus {
    border-color: #2590ff;
    box-shadow: 0 0 0 0.2rem rgba(37, 144, 255, 0.25);
    background: #ffffff;
}

/* プログレスバー風文字数表示 */
.char-progress {
    height: 4px;
    background: #e9ecef;
    border-radius: 2px;
    margin-top: 5px;
    overflow: hidden;
}

.char-progress-bar {
    height: 100%;
    background: linear-gradient(90deg, #28a745, #ffc107, #dc3545);
    border-radius: 2px;
    transition: width 0.3s ease;
}
```

### 実装優先順位
1. **既存フィールド削除** （データベース・コード整理）
2. **レスポンシブテキストエリア改善** （PC使いやすさ向上）
3. **全体レイアウト改善** （視覚的改善）
4. **インタラクティブ機能追加** （UX向上）

---

**統合指示:** 古いフィールド削除作業と同時に、編集画面のデザイン改善も実装してください。特にPC表示時のテキストエリアサイズとレスポンシブ対応を重視してください。