# Claude Code向け 学生個別ページ修正 指示書

## 🚨 修正が必要な問題

### 問題1: デザイン崩れ
- 学生個別ページのレイアウトが正しく表示されない
- CSS・Bootstrap クラスの適用不備

### 問題2: 編集時のエラー
- 「アクセスが集中しています。しばらくしてからお試しください。」
- レートリミット・認証・CSRF関連のエラーの可能性

## 🔧 修正指示

### 1. デザイン崩れの修正

#### A. Bootstrap・CSS読み込み確認
```html
<!-- views/student-profile.ejs と views/student-edit.ejs で確認 -->
<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><%= student.name %> - プロフィール</title>
    
    <!-- Bootstrap CSS（CDN） -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    
    <!-- Bootstrap Icons -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.7.2/font/bootstrap-icons.css">
    
    <!-- カスタムCSS -->
    <link href="/css/style.css" rel="stylesheet">
    
    <!-- SEO設定 -->
    <meta name="robots" content="noindex, nofollow">
</head>
```

#### B. レスポンシブ・レイアウト修正
```html
<!-- student-profile.ejs の修正版 -->
<div class="container-fluid mt-4">
    <div class="row justify-content-center">
        <div class="col-12 col-lg-10">
            <!-- パンくずナビ -->
            <nav aria-label="breadcrumb" class="mb-4">
                <ol class="breadcrumb">
                    <li class="breadcrumb-item"><a href="/" class="text-decoration-none">ホーム</a></li>
                    <li class="breadcrumb-item active" aria-current="page"><%= student.name %>のプロフィール</li>
                </ol>
            </nav>
            
            <div class="row">
                <!-- 基本情報カラム -->
                <div class="col-12 col-md-4 mb-4">
                    <div class="card h-100 shadow-sm">
                        <div class="card-body text-center">
                            <div class="mb-3">
                                <img src="<%= student.photo_path || '/images/default-avatar.png' %>" 
                                     alt="<%= student.name %>" 
                                     class="rounded-circle border"
                                     style="width: 180px; height: 180px; object-fit: cover;">
                            </div>
                            
                            <h4 class="card-title text-primary mb-2"><%= student.name %></h4>
                            
                            <div class="mb-3">
                                <p class="mb-1 text-muted">
                                    <i class="bi bi-building me-1"></i>
                                    <%= student.school_name %>
                                </p>
                                <p class="mb-1 text-muted">
                                    <i class="bi bi-person-badge me-1"></i>
                                    <%= student.grade %>
                                </p>
                                <% if (student.group_name) { %>
                                    <span class="badge bg-primary fs-6 px-3 py-2">
                                        <i class="bi bi-people-fill me-1"></i>
                                        <%= student.group_name %>班
                                    </span>
                                <% } %>
                            </div>
                            
                            <div class="d-grid gap-2">
                                <a href="/student/<%= student.id %>/edit" class="btn btn-outline-primary">
                                    <i class="bi bi-pencil-square me-1"></i>
                                    プロフィール編集
                                </a>
                                <a href="/" class="btn btn-outline-secondary btn-sm">
                                    <i class="bi bi-arrow-left me-1"></i>
                                    一覧に戻る
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- 詳細情報カラム -->
                <div class="col-12 col-md-8">
                    <div class="card shadow-sm">
                        <div class="card-header bg-light">
                            <h5 class="mb-0 text-primary">
                                <i class="bi bi-person-lines-fill me-2"></i>
                                詳細プロフィール
                            </h5>
                        </div>
                        <div class="card-body">
                            <!-- 各セクション -->
                            <div class="profile-sections">
                                <!-- 研修参加について -->
                                <div class="profile-section mb-4">
                                    <h6 class="section-title border-start border-primary border-3 ps-3 mb-3">
                                        <i class="bi bi-mortarboard-fill text-primary me-2"></i>
                                        研修参加について
                                    </h6>
                                    <div class="content-box p-3 bg-light rounded">
                                        <% if (student.participation_comment && student.participation_comment.trim()) { %>
                                            <p class="mb-0 lh-lg"><%= student.participation_comment %></p>
                                        <% } else { %>
                                            <p class="text-muted fst-italic mb-0">
                                                <i class="bi bi-info-circle me-1"></i>
                                                まだ入力されていません
                                            </p>
                                        <% } %>
                                    </div>
                                </div>
                                
                                <!-- 趣味・自己PR -->
                                <div class="profile-section mb-4">
                                    <h6 class="section-title border-start border-success border-3 ps-3 mb-3">
                                        <i class="bi bi-heart-fill text-success me-2"></i>
                                        趣味・自己PR
                                    </h6>
                                    <div class="content-box p-3 bg-light rounded">
                                        <% if (student.hobby_selfpr && student.hobby_selfpr.trim()) { %>
                                            <p class="mb-0 lh-lg"><%= student.hobby_selfpr %></p>
                                        <% } else { %>
                                            <p class="text-muted fst-italic mb-0">
                                                <i class="bi bi-info-circle me-1"></i>
                                                まだ入力されていません
                                            </p>
                                        <% } %>
                                    </div>
                                </div>
                                
                                <!-- 事前学習課題 -->
                                <div class="profile-section mb-4">
                                    <h6 class="section-title border-start border-warning border-3 ps-3 mb-3">
                                        <i class="bi bi-book-fill text-warning me-2"></i>
                                        事前学習課題
                                    </h6>
                                    <div class="content-box p-3 bg-light rounded">
                                        <% if (student.prestudy_task && student.prestudy_task.trim()) { %>
                                            <p class="mb-0 lh-lg"><%= student.prestudy_task %></p>
                                        <% } else { %>
                                            <p class="text-muted fst-italic mb-0">
                                                <i class="bi bi-info-circle me-1"></i>
                                                まだ入力されていません
                                            </p>
                                        <% } %>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- 更新成功アラート -->
    <% if (typeof query !== 'undefined' && query.updated === 'true') { %>
        <div class="position-fixed top-0 end-0 p-3" style="z-index: 1050;">
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                <i class="bi bi-check-circle-fill me-2"></i>
                プロフィールを更新しました！
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        </div>
    <% } %>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
```

### 2. 編集エラーの修正

#### A. 認証・セッション確認
```javascript
// routes/student.js の編集ページアクセス制御
router.get('/:id/edit', async (req, res) => {
    try {
        const studentId = req.params.id;
        
        // 学生情報取得
        const student = await db.get(`
            SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
            FROM participants p 
            LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
            WHERE p.id = ? AND p.type = 'student'
        `, [studentId]);
        
        if (!student) {
            return res.status(404).render('error', { message: '学生が見つかりません' });
        }
        
        // 編集権限チェック（一旦コメントアウトして動作確認）
        // if (req.session.student_id !== parseInt(studentId)) {
        //     return res.status(403).render('error', { message: 'この学生のプロフィールを編集する権限がありません' });
        // }
        
        res.render('student-edit', { student, error: null });
    } catch (error) {
        console.error('Student edit page error:', error);
        res.status(500).render('error', { message: 'システムエラーが発生しました' });
    }
});
```

#### B. 更新処理の修正（CSRF・レートリミット対応）
```javascript
// routes/student.js の更新処理修正
router.post('/:id/update', async (req, res) => {
    try {
        const studentId = parseInt(req.params.id);
        const { participation_comment, hobby_selfpr, prestudy_task } = req.body;
        
        // 入力値検証
        if (!participation_comment && !hobby_selfpr && !prestudy_task) {
            return res.render('student-edit', { 
                student: await getStudentData(studentId),
                error: '少なくとも1つの項目を入力してください' 
            });
        }
        
        // 文字数制限
        const maxLength = 1000;
        if (participation_comment && participation_comment.length > maxLength) {
            return res.render('student-edit', { 
                student: await getStudentData(studentId),
                error: '入力内容が長すぎます（1000文字以内）' 
            });
        }
        
        // データベース更新
        const existingProfile = await db.get(
            'SELECT * FROM student_profiles WHERE participant_id = ?', 
            [studentId]
        );
        
        if (existingProfile) {
            // 更新
            await db.run(`
                UPDATE student_profiles 
                SET participation_comment = ?, 
                    hobby_selfpr = ?, 
                    prestudy_task = ?, 
                    updated_at = CURRENT_TIMESTAMP
                WHERE participant_id = ?
            `, [
                participation_comment || '', 
                hobby_selfpr || '', 
                prestudy_task || '', 
                studentId
            ]);
        } else {
            // 新規作成
            await db.run(`
                INSERT INTO student_profiles 
                (participant_id, participation_comment, hobby_selfpr, prestudy_task, updated_at)
                VALUES (?, ?, ?, ?, CURRENT_TIMESTAMP)
            `, [
                studentId, 
                participation_comment || '', 
                hobby_selfpr || '', 
                prestudy_task || ''
            ]);
        }
        
        // 成功時のリダイレクト
        res.redirect(`/student/${studentId}?updated=true`);
        
    } catch (error) {
        console.error('Student update error:', error);
        
        // エラー時の処理
        const student = await getStudentData(req.params.id);
        res.render('student-edit', { 
            student, 
            error: 'プロフィールの更新に失敗しました。しばらくしてからお試しください。' 
        });
    }
});

// ヘルパー関数
async function getStudentData(studentId) {
    return await db.get(`
        SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
        FROM participants p 
        LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
        WHERE p.id = ? AND p.type = 'student'
    `, [studentId]);
}
```

#### C. エラーハンドリング強化
```html
<!-- student-edit.ejs の修正版（エラー表示追加） -->
<div class="card">
    <div class="card-header bg-light">
        <h5 class="mb-0 text-primary">
            <i class="bi bi-pencil-square me-2"></i>
            プロフィール編集
        </h5>
    </div>
    <div class="card-body">
        <!-- エラーメッセージ表示 -->
        <% if (typeof error !== 'undefined' && error) { %>
            <div class="alert alert-danger" role="alert">
                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                <%= error %>
            </div>
        <% } %>
        
        <form method="POST" action="/student/<%= student.id %>/update" id="profileForm">
            <!-- フォーム内容 -->
            <div class="mb-4">
                <label for="participation_comment" class="form-label fw-bold">
                    <i class="bi bi-mortarboard-fill text-primary me-2"></i>
                    研修参加について
                </label>
                <textarea class="form-control" id="participation_comment" 
                          name="participation_comment" rows="4" maxlength="1000"
                          placeholder="研修に参加する目的や意気込みを教えてください（1000文字以内）"><%= student.participation_comment || '' %></textarea>
                <div class="form-text">
                    <span class="text-muted">文字数: </span>
                    <span id="participation_count">0</span>/1000
                </div>
            </div>
            
            <div class="mb-4">
                <label for="hobby_selfpr" class="form-label fw-bold">
                    <i class="bi bi-heart-fill text-success me-2"></i>
                    趣味・自己PR
                </label>
                <textarea class="form-control" id="hobby_selfpr" 
                          name="hobby_selfpr" rows="4" maxlength="1000"
                          placeholder="趣味や特技、自己PRを教えてください（1000文字以内）"><%= student.hobby_selfpr || '' %></textarea>
                <div class="form-text">
                    <span class="text-muted">文字数: </span>
                    <span id="hobby_count">0</span>/1000
                </div>
            </div>
            
            <div class="mb-4">
                <label for="prestudy_task" class="form-label fw-bold">
                    <i class="bi bi-book-fill text-warning me-2"></i>
                    事前学習課題
                </label>
                <textarea class="form-control" id="prestudy_task" 
                          name="prestudy_task" rows="4" maxlength="1000"
                          placeholder="事前学習の成果や学んだことを教えてください（1000文字以内）"><%= student.prestudy_task || '' %></textarea>
                <div class="form-text">
                    <span class="text-muted">文字数: </span>
                    <span id="prestudy_count">0</span>/1000
                </div>
            </div>
            
            <!-- 送信ボタン -->
            <div class="d-flex gap-2 justify-content-between">
                <a href="/student/<%= student.id %>" class="btn btn-outline-secondary">
                    <i class="bi bi-arrow-left me-1"></i>
                    戻る
                </a>
                <button type="submit" class="btn btn-primary" id="submitBtn">
                    <i class="bi bi-check-lg me-1"></i>
                    更新する
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// 文字数カウンター
function setupCharacterCounter(textareaId, counterId) {
    const textarea = document.getElementById(textareaId);
    const counter = document.getElementById(counterId);
    
    function updateCounter() {
        const count = textarea.value.length;
        counter.textContent = count;
        
        if (count > 1000) {
            counter.parentElement.classList.add('text-danger');
        } else {
            counter.parentElement.classList.remove('text-danger');
        }
    }
    
    textarea.addEventListener('input', updateCounter);
    updateCounter(); // 初期表示
}

// ページ読み込み時に文字数カウンター設定
document.addEventListener('DOMContentLoaded', function() {
    setupCharacterCounter('participation_comment', 'participation_count');
    setupCharacterCounter('hobby_selfpr', 'hobby_count');
    setupCharacterCounter('prestudy_task', 'prestudy_count');
});

// フォーム送信時の重複防止
document.getElementById('profileForm').addEventListener('submit', function() {
    const submitBtn = document.getElementById('submitBtn');
    submitBtn.disabled = true;
    submitBtn.innerHTML = '<i class="bi bi-hourglass-split me-1"></i>更新中...';
});
</script>
```

### 3. server.js でのエラーハンドリング追加
```javascript
// server.js にエラーハンドリング追加
app.use('/student', (req, res, next) => {
    // 学生ルートのレートリミット緩和
    req.skipRateLimit = true;
    next();
});

// 学生ルート追加
app.use('/student', studentRoutes);

// 404エラーハンドリング
app.use((req, res) => {
    res.status(404).render('error', { message: 'ページが見つかりません' });
});

// 500エラーハンドリング  
app.use((error, req, res, next) => {
    console.error('Server Error:', error);
    res.status(500).render('error', { message: 'システムエラーが発生しました' });
});
```

## ✅ 修正チェックリスト

### デザイン修正
- [ ] Bootstrap CSS/JS正常読み込み
- [ ] レスポンシブレイアウト動作
- [ ] アイコン・色彩統一
- [ ] 文字数カウンター表示

### エラー修正
- [ ] 編集ページアクセス可能
- [ ] フォーム送信成功
- [ ] エラーメッセージ適切表示
- [ ] 重複送信防止

## 🧪 テスト項目

1. **デザインテスト**
   - [ ] PC・スマホでの表示確認
   - [ ] 各要素の配置・色彩確認

2. **機能テスト**  
   - [ ] 編集ページアクセス
   - [ ] テキスト入力・更新
   - [ ] 成功・エラー時の表示

---

**修正優先順位:**
1. 編集エラー修正（機能重視）
2. デザイン崩れ修正（見た目改善）
3. UX向上（文字数カウンター等）