# Claude Code向け 学生個別プロフィールページ作成 指示書

## 🎯 機能概要
参加学生の詳細情報を表示・編集できる個別ページを作成します。

### 表示・編集項目
- **基本情報（管理者設定）**: 名前・学校名・学年・班・顔写真
- **学生編集可能項目（新規追加）**: 
  1. 研修参加に関して
  2. 趣味・自己PR  
  3. 事前学習課題

## 📋 実装要件

### 1. データベース拡張

#### student_profilesテーブル拡張
```sql
-- 既存のカラムに新しい項目を追加
ALTER TABLE student_profiles ADD COLUMN participation_comment TEXT;
ALTER TABLE student_profiles ADD COLUMN hobby_selfpr TEXT;  
ALTER TABLE student_profiles ADD COLUMN prestudy_task TEXT;

-- 既存データがある場合のデフォルト値設定
UPDATE student_profiles SET 
    participation_comment = '',
    hobby_selfpr = '',
    prestudy_task = ''
WHERE participation_comment IS NULL;
```

### 2. URL構造・ルーティング

#### 新規ルート追加
```
/student/:id          # 学生個別プロフィール表示ページ
/student/:id/edit     # 学生個別編集ページ
```

#### routes/student.js の作成
```javascript
const express = require('express');
const router = express.Router();

// 学生プロフィール表示
router.get('/:id', async (req, res) => {
    try {
        const studentId = req.params.id;
        
        // 学生基本情報取得
        const student = await db.get(`
            SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
            FROM participants p 
            LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
            WHERE p.id = ? AND p.type = 'student'
        `, [studentId]);
        
        if (!student) {
            return res.status(404).render('error', { message: '学生が見つかりません' });
        }
        
        res.render('student-profile', { student });
    } catch (error) {
        res.status(500).render('error', { message: error.message });
    }
});

// 学生編集ページ表示
router.get('/:id/edit', async (req, res) => {
    try {
        const studentId = req.params.id;
        const student = await db.get(`
            SELECT p.*, sp.participation_comment, sp.hobby_selfpr, sp.prestudy_task
            FROM participants p 
            LEFT JOIN student_profiles sp ON p.id = sp.participant_id 
            WHERE p.id = ? AND p.type = 'student'
        `, [studentId]);
        
        if (!student) {
            return res.status(404).render('error', { message: '学生が見つかりません' });
        }
        
        res.render('student-edit', { student });
    } catch (error) {
        res.status(500).render('error', { message: error.message });
    }
});

// 学生プロフィール更新
router.post('/:id/update', async (req, res) => {
    try {
        const studentId = req.params.id;
        const { participation_comment, hobby_selfpr, prestudy_task } = req.body;
        
        // student_profilesテーブルに存在確認・作成
        const existingProfile = await db.get(
            'SELECT * FROM student_profiles WHERE participant_id = ?', 
            [studentId]
        );
        
        if (existingProfile) {
            // 更新
            await db.run(`
                UPDATE student_profiles 
                SET participation_comment = ?, hobby_selfpr = ?, prestudy_task = ?, updated_at = CURRENT_TIMESTAMP
                WHERE participant_id = ?
            `, [participation_comment, hobby_selfpr, prestudy_task, studentId]);
        } else {
            // 新規作成
            await db.run(`
                INSERT INTO student_profiles (participant_id, participation_comment, hobby_selfpr, prestudy_task)
                VALUES (?, ?, ?, ?)
            `, [studentId, participation_comment, hobby_selfpr, prestudy_task]);
        }
        
        res.redirect(`/student/${studentId}?updated=true`);
    } catch (error) {
        res.status(500).render('error', { message: error.message });
    }
});

module.exports = router;
```

### 3. テンプレート作成

#### views/student-profile.ejs
```html
<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><%= student.name %> - プロフィール | 2590地区インターアクト研修旅行</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="/css/style.css" rel="stylesheet">
    <meta name="robots" content="noindex, nofollow">
</head>
<body>
    <div class="container mt-4">
        <!-- ヘッダー -->
        <div class="row mb-4">
            <div class="col-12">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="/">ホーム</a></li>
                        <li class="breadcrumb-item active"><%= student.name %>のプロフィール</li>
                    </ol>
                </nav>
            </div>
        </div>
        
        <!-- 学生プロフィール -->
        <div class="row">
            <div class="col-lg-4">
                <!-- 基本情報カード -->
                <div class="card">
                    <div class="card-body text-center">
                        <img src="<%= student.photo_path || '/images/default-avatar.png' %>" 
                             alt="<%= student.name %>" 
                             class="student-photo rounded-circle mb-3"
                             style="width: 200px; height: 200px; object-fit: cover;">
                        
                        <h4 class="card-title"><%= student.name %></h4>
                        <p class="text-muted"><%= student.school_name %></p>
                        <p class="text-muted"><%= student.grade %></p>
                        
                        <% if (student.group_name) { %>
                            <span class="badge bg-primary fs-6 mb-3"><%= student.group_name %>班</span>
                        <% } %>
                        
                        <!-- 編集ボタン -->
                        <div class="d-grid">
                            <a href="/student/<%= student.id %>/edit" class="btn btn-outline-primary">
                                プロフィール編集
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-8">
                <!-- 詳細情報 -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">詳細プロフィール</h5>
                    </div>
                    <div class="card-body">
                        <!-- 研修参加に関して -->
                        <div class="profile-section mb-4">
                            <h6 class="section-title">
                                <i class="bi bi-mortarboard-fill text-primary me-2"></i>
                                研修参加に関して
                            </h6>
                            <div class="content-box">
                                <% if (student.participation_comment && student.participation_comment.trim()) { %>
                                    <p><%= student.participation_comment %></p>
                                <% } else { %>
                                    <p class="text-muted">まだ入力されていません</p>
                                <% } %>
                            </div>
                        </div>
                        
                        <!-- 趣味・自己PR -->
                        <div class="profile-section mb-4">
                            <h6 class="section-title">
                                <i class="bi bi-person-heart text-success me-2"></i>
                                趣味・自己PR
                            </h6>
                            <div class="content-box">
                                <% if (student.hobby_selfpr && student.hobby_selfpr.trim()) { %>
                                    <p><%= student.hobby_selfpr %></p>
                                <% } else { %>
                                    <p class="text-muted">まだ入力されていません</p>
                                <% } %>
                            </div>
                        </div>
                        
                        <!-- 事前学習課題 -->
                        <div class="profile-section mb-4">
                            <h6 class="section-title">
                                <i class="bi bi-book-fill text-warning me-2"></i>
                                事前学習課題
                            </h6>
                            <div class="content-box">
                                <% if (student.prestudy_task && student.prestudy_task.trim()) { %>
                                    <p><%= student.prestudy_task %></p>
                                <% } else { %>
                                    <p class="text-muted">まだ入力されていません</p>
                                <% } %>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- 更新完了メッセージ -->
        <% if (typeof query !== 'undefined' && query.updated) { %>
            <div class="alert alert-success alert-dismissible fade show" role="alert">
                プロフィールを更新しました！
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <% } %>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
```

#### views/student-edit.ejs
```html
<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><%= student.name %> - プロフィール編集 | 2590地区インターアクト研修旅行</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="/css/style.css" rel="stylesheet">
    <meta name="robots" content="noindex, nofollow">
</head>
<body>
    <div class="container mt-4">
        <!-- ヘッダー -->
        <div class="row mb-4">
            <div class="col-12">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb">
                        <li class="breadcrumb-item"><a href="/">ホーム</a></li>
                        <li class="breadcrumb-item"><a href="/student/<%= student.id %>"><%= student.name %></a></li>
                        <li class="breadcrumb-item active">プロフィール編集</li>
                    </ol>
                </nav>
            </div>
        </div>
        
        <div class="row">
            <div class="col-lg-4">
                <!-- 基本情報表示 -->
                <div class="card">
                    <div class="card-body text-center">
                        <img src="<%= student.photo_path || '/images/default-avatar.png' %>" 
                             alt="<%= student.name %>" 
                             class="student-photo rounded-circle mb-3"
                             style="width: 150px; height: 150px; object-fit: cover;">
                        <h5><%= student.name %></h5>
                        <p class="text-muted">
                            <%= student.school_name %><br>
                            <%= student.grade %><br>
                            <% if (student.group_name) { %>
                                <span class="badge bg-primary"><%= student.group_name %>班</span>
                            <% } %>
                        </p>
                    </div>
                </div>
            </div>
            
            <div class="col-lg-8">
                <!-- 編集フォーム -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">プロフィール編集</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" action="/student/<%= student.id %>/update">
                            <!-- 研修参加に関して -->
                            <div class="mb-4">
                                <label for="participation_comment" class="form-label">
                                    <i class="bi bi-mortarboard-fill text-primary me-2"></i>
                                    研修参加に関して
                                </label>
                                <textarea class="form-control" id="participation_comment" 
                                          name="participation_comment" rows="4"
                                          placeholder="研修に参加する目的や意気込みを教えてください"><%= student.participation_comment || '' %></textarea>
                                <div class="form-text">研修に対する思いや目標を自由に書いてください</div>
                            </div>
                            
                            <!-- 趣味・自己PR -->
                            <div class="mb-4">
                                <label for="hobby_selfpr" class="form-label">
                                    <i class="bi bi-person-heart text-success me-2"></i>
                                    趣味・自己PR
                                </label>
                                <textarea class="form-control" id="hobby_selfpr" 
                                          name="hobby_selfpr" rows="4"
                                          placeholder="趣味や特技、自己PRを教えてください"><%= student.hobby_selfpr || '' %></textarea>
                                <div class="form-text">あなたの個性や魅力を自由にアピールしてください</div>
                            </div>
                            
                            <!-- 事前学習課題 -->
                            <div class="mb-4">
                                <label for="prestudy_task" class="form-label">
                                    <i class="bi bi-book-fill text-warning me-2"></i>
                                    事前学習課題
                                </label>
                                <textarea class="form-control" id="prestudy_task" 
                                          name="prestudy_task" rows="4"
                                          placeholder="事前学習の成果や学んだことを教えてください"><%= student.prestudy_task || '' %></textarea>
                                <div class="form-text">事前学習で調べたことや感想を記入してください</div>
                            </div>
                            
                            <!-- ボタン -->
                            <div class="d-flex gap-2">
                                <button type="submit" class="btn btn-primary">
                                    <i class="bi bi-check-lg me-1"></i>
                                    更新する
                                </button>
                                <a href="/student/<%= student.id %>" class="btn btn-secondary">
                                    <i class="bi bi-arrow-left me-1"></i>
                                    戻る
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
```

### 4. トップページの修正

#### views/index.ejs の参加学生セクション修正
```html
<!-- 参加学生セクション -->
<div class="participant-category">
    <h3>参加学生</h3>
    <div class="participants-grid">
        <% students.forEach(student => { %>
            <div class="participant-card" data-type="student">
                <a href="/student/<%= student.id %>" class="student-link">
                    <img src="<%= student.photo_path || '/images/default-avatar.png' %>" alt="<%= student.name %>">
                    <h4><%= student.name %></h4>
                    <p class="school"><%= student.school_name %></p>
                    <p class="grade"><%= student.grade %></p>
                    <span class="group-badge group-<%= student.group_name %>"><%= student.group_name %>班</span>
                    <div class="view-detail">詳細を見る</div>
                </a>
            </div>
        <% }) %>
    </div>
</div>
```

### 5. CSS スタイリング

#### public/css/style.css に追加
```css
/* 学生個別ページ */
.student-link {
    text-decoration: none;
    color: inherit;
    display: block;
    transition: transform 0.2s;
}

.student-link:hover {
    transform: translateY(-5px);
    text-decoration: none;
    color: inherit;
}

.view-detail {
    background: linear-gradient(135deg, #2590ff, #1e7dd8);
    color: white;
    text-align: center;
    padding: 8px;
    border-radius: 0 0 10px 10px;
    font-size: 0.9em;
    opacity: 0;
    transition: opacity 0.3s;
}

.participant-card:hover .view-detail {
    opacity: 1;
}

/* プロフィールページ */
.profile-section {
    border-left: 4px solid #2590ff;
    padding-left: 15px;
}

.section-title {
    color: #2590ff;
    font-weight: 600;
    margin-bottom: 10px;
}

.content-box {
    background-color: #f8f9fa;
    padding: 15px;
    border-radius: 8px;
    min-height: 60px;
}

.content-box p {
    margin-bottom: 0;
    line-height: 1.6;
}

/* 編集フォーム */
.form-label {
    font-weight: 600;
    color: #333;
}

.form-text {
    color: #666;
    font-size: 0.9em;
}

/* レスポンシブ対応 */
@media (max-width: 768px) {
    .student-photo {
        width: 120px !important;
        height: 120px !important;
    }
    
    .profile-section {
        margin-bottom: 20px !important;
    }
}
```

### 6. server.js にルート追加
```javascript
// server.js に以下を追加
const studentRoutes = require('./routes/student');
app.use('/student', studentRoutes);
```

## ✅ 実装チェックリスト

### データベース
- [ ] student_profilesテーブルに新規カラム追加
- [ ] 既存データの整合性確認

### ルーティング
- [ ] /student/:id ルート作成
- [ ] /student/:id/edit ルート作成
- [ ] POST /student/:id/update 実装

### テンプレート
- [ ] student-profile.ejs 作成
- [ ] student-edit.ejs 作成
- [ ] index.ejs の学生カード修正

### CSS・UI
- [ ] 個別ページのデザイン
- [ ] ホバー効果・トランジション
- [ ] レスポンシブ対応

## 🧪 テスト項目

### 基本動作テスト
- [ ] トップページから学生個別ページへのリンク
- [ ] 個別ページでの詳細情報表示
- [ ] 編集ページでの項目入力・更新
- [ ] 更新後の表示確認

### データ整合性テスト
- [ ] 新規学生の個別ページ表示
- [ ] 既存学生データとの互換性
- [ ] 空データ時の適切な表示

### UX テスト
- [ ] パンくずナビゲーション
- [ ] 編集完了時のフィードバック
- [ ] モバイル表示での操作性

## 📝 完了報告

実装完了時に以下を報告してください：
1. **データベース拡張の完了確認**
2. **各学生の個別ページ表示テスト結果**  
3. **編集機能の動作確認結果**
4. **既存機能への影響確認**

---

**実装順序推奨:**
1. データベース拡張
2. ルーティング・API実装  
3. テンプレート作成
4. CSS・UI調整
5. トップページリンク修正