# Claude Code向け 引率者追加・必要情報一括削除機能 追加指示書

## 🎯 機能追加概要

### 1. 参加者に「引率者」カテゴリ追加
- **目的:** 学生・同行者に加えて「引率者」を管理
- **権限:** 管理者のみ追加・編集（個別パスワード不要）
- **表示:** 参加者一覧で引率者として区別表示

### 2. 必要情報の一括削除機能
- **目的:** 必要情報ページでもチェックボックス一括削除を可能に
- **機能:** お知らせ等と同様の一括削除UI追加

## 📋 機能1: 引率者カテゴリ追加

### データベース変更

#### participantsテーブルの型変更
```sql
-- 現在の type カラムを拡張
-- 既存: 'student', 'companion' 
-- 変更後: 'student', 'companion', 'chaperone'

-- ALTER文（既存データ保持）
ALTER TABLE participants CHECK (type IN ('student', 'companion', 'chaperone'));
```

### 管理画面UI変更

#### 参加者追加・編集フォーム
```html
<!-- 参加者タイプ選択 -->
<div class="form-group">
    <label for="participantType">参加者タイプ</label>
    <select class="form-control" id="participantType" name="type" required>
        <option value="">選択してください</option>
        <option value="student">参加学生</option>
        <option value="companion">同行者</option>
        <option value="chaperone">引率者</option>
    </select>
</div>

<!-- 学生専用フィールド（条件表示） -->
<div id="studentFields" style="display: none;">
    <div class="form-group">
        <label for="schoolName">学校名</label>
        <input type="text" class="form-control" id="schoolName" name="school_name">
    </div>
    <div class="form-group">
        <label for="grade">学年</label>
        <input type="text" class="form-control" id="grade" name="grade">
    </div>
    <div class="form-group">
        <label for="groupName">班名</label>
        <select class="form-control" id="groupName" name="group_name">
            <option value="">選択してください</option>
            <option value="A">A班</option>
            <option value="B">B班</option>
            <option value="C">C班</option>
            <option value="D">D班</option>
            <option value="E">E班</option>
            <option value="F">F班</option>
        </select>
    </div>
    <div class="form-group">
        <label for="password">個別パスワード</label>
        <input type="text" class="form-control" id="password" name="password">
    </div>
</div>
```

#### JavaScript（条件表示制御）
```javascript
// 参加者タイプに応じたフィールド表示制御
document.getElementById('participantType').addEventListener('change', function() {
    const studentFields = document.getElementById('studentFields');
    const isStudent = this.value === 'student';
    
    studentFields.style.display = isStudent ? 'block' : 'none';
    
    // 非表示時は値をクリア
    if (!isStudent) {
        studentFields.querySelectorAll('input, select').forEach(field => {
            field.value = '';
        });
    }
});
```

### 公開サイト表示変更

#### 参加者一覧のセクション分け
```html
<!-- 公開サイトの参加者表示 -->
<div class="participants-section">
    <!-- 参加学生セクション -->
    <div class="participant-category">
        <h3>参加学生</h3>
        <div class="participants-grid">
            <% students.forEach(student => { %>
                <div class="participant-card" data-type="student">
                    <img src="<%= student.photo_path || '/images/default-avatar.png' %>" alt="<%= student.name %>">
                    <h4><%= student.name %></h4>
                    <p class="school"><%= student.school_name %></p>
                    <p class="grade"><%= student.grade %></p>
                    <span class="group-badge group-<%= student.group_name %>"><%= student.group_name %>班</span>
                </div>
            <% }) %>
        </div>
    </div>
    
    <!-- 同行者セクション -->
    <div class="participant-category">
        <h3>同行者</h3>
        <div class="participants-list">
            <% companions.forEach(companion => { %>
                <div class="participant-item" data-type="companion">
                    <img src="<%= companion.photo_path || '/images/default-avatar.png' %>" alt="<%= companion.name %>">
                    <span class="name"><%= companion.name %></span>
                </div>
            <% }) %>
        </div>
    </div>
    
    <!-- 引率者セクション（新規追加） -->
    <div class="participant-category">
        <h3>引率者</h3>
        <div class="participants-list">
            <% chaperones.forEach(chaperone => { %>
                <div class="participant-item" data-type="chaperone">
                    <img src="<%= chaperone.photo_path || '/images/default-avatar.png' %>" alt="<%= chaperone.name %>">
                    <span class="name"><%= chaperone.name %></span>
                    <span class="role-badge">引率者</span>
                </div>
            <% }) %>
        </div>
    </div>
</div>
```

### バックエンド変更

#### ルーターの修正（routes/public.js）
```javascript
// 参加者データを type で分類して取得
router.get('/api/participants', async (req, res) => {
    try {
        const participants = await db.all('SELECT * FROM participants ORDER BY type, name');
        
        const categorized = {
            students: participants.filter(p => p.type === 'student'),
            companions: participants.filter(p => p.type === 'companion'),
            chaperones: participants.filter(p => p.type === 'chaperone')
        };
        
        res.json(categorized);
    } catch (error) {
        res.status(500).json({ error: error.message });
    }
});
```

### CSS スタイリング
```css
/* 引率者専用スタイル */
.participant-item[data-type="chaperone"] {
    border-left: 4px solid #28a745; /* 緑色で区別 */
}

.role-badge {
    background-color: #28a745;
    color: white;
    padding: 2px 8px;
    border-radius: 12px;
    font-size: 0.8em;
    margin-left: 10px;
}

.participant-category {
    margin-bottom: 30px;
}

.participant-category h3 {
    color: #2590ff;
    border-bottom: 2px solid #2590ff;
    padding-bottom: 5px;
    margin-bottom: 20px;
}
```

## 📋 機能2: 必要情報の一括削除

### 管理画面への一括削除機能追加

#### UI追加（管理画面の必要情報セクション）
```html
<!-- 必要情報管理セクションに追加 -->
<div class="information-management">
    <h3>必要情報管理</h3>
    
    <!-- 一括削除UI -->
    <div class="bulk-actions">
        <label>
            <input type="checkbox" id="selectAllInfo" />
            全選択
        </label>
        <button id="bulkDeleteInfoBtn" class="btn btn-danger" disabled>
            選択したアイテムを削除 (<span id="selectedInfoCount">0</span>件)
        </button>
    </div>
    
    <!-- 情報一覧テーブル -->
    <table class="table">
        <thead>
            <tr>
                <th width="50px">
                    <input type="checkbox" id="selectAllInfoInTable" />
                </th>
                <th>カテゴリ</th>
                <th>タイトル</th>
                <th>内容</th>
                <th>操作</th>
            </tr>
        </thead>
        <tbody id="informationTableBody">
            <!-- 動的に生成 -->
        </tbody>
    </table>
</div>
```

#### JavaScript（一括削除機能）
```javascript
// 必要情報の一括削除機能
function initInformationBulkDelete() {
    // 全選択機能
    document.getElementById('selectAllInfo').addEventListener('change', function() {
        const checkboxes = document.querySelectorAll('.info-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = this.checked;
        });
        updateInfoBulkDeleteButton();
    });
    
    // 一括削除ボタンイベント
    document.getElementById('bulkDeleteInfoBtn').addEventListener('click', function() {
        const checkedBoxes = document.querySelectorAll('.info-checkbox:checked');
        const ids = Array.from(checkedBoxes).map(checkbox => checkbox.value);
        
        if (ids.length === 0) {
            alert('削除するアイテムを選択してください');
            return;
        }
        
        const confirmMessage = `選択した${ids.length}件の必要情報を削除しますか？`;
        
        if (confirm(confirmMessage)) {
            bulkDeleteInformation(ids);
        }
    });
}

// 必要情報一括削除API呼び出し
function bulkDeleteInformation(ids) {
    fetch('/admin/api/bulk-delete', {
        method: 'DELETE',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            type: 'information',
            ids: ids
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert(`${data.deletedCount}件の必要情報を削除しました`);
            loadInformationList(); // テーブル再読み込み
        } else {
            alert('削除に失敗しました: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('削除処理でエラーが発生しました');
    });
}
```

#### バックエンド修正（一括削除対応）
```javascript
// routes/admin.js の bulk-delete エンドポイントに 'information' を追加
router.delete('/api/bulk-delete', async (req, res) => {
    try {
        const { type, ids } = req.body;
        
        // 許可されたテーブルに 'information' を追加
        const allowedTypes = ['news', 'schedules', 'participants', 'information'];
        
        if (!allowedTypes.includes(type)) {
            return res.status(400).json({ 
                success: false, 
                message: '無効なテーブル名です' 
            });
        }
        
        // 以降は既存の処理と同じ
        // ...
    } catch (error) {
        console.error('Bulk delete error:', error);
        res.status(500).json({ success: false, error: error.message });
    }
});
```

## ✅ 実装チェックリスト

### 引率者機能
- [ ] データベース型制約の更新
- [ ] 管理画面での引率者追加・編集
- [ ] 学生専用フィールドの条件表示
- [ ] 公開サイトでの引率者セクション表示
- [ ] 引率者にはパスワード設定なし

### 必要情報一括削除
- [ ] チェックボックスUI追加
- [ ] 全選択機能実装
- [ ] 一括削除API対応
- [ ] 削除確認・完了メッセージ

## 🧪 テスト項目

### 引率者機能テスト
1. **管理画面テスト**
   - [ ] 引率者の新規追加
   - [ ] 学生選択時のみ学校情報フィールド表示
   - [ ] 引率者には学校情報・パスワード不要

2. **公開サイトテスト**
   - [ ] 参加者一覧で3カテゴリ表示
   - [ ] 引率者の区別表示

### 必要情報一括削除テスト
- [ ] 複数選択→一括削除実行
- [ ] 全選択機能の動作
- [ ] 削除完了後のリスト更新

## 📝 完了報告

実装完了時に報告してください：
1. **引率者機能の動作確認結果**
2. **必要情報一括削除の動作確認**
3. **既存機能への影響がないことの確認**
4. **UIの統一性確認**

---

**実装順序推奨:** 
1. 引率者機能（データベース→管理画面→公開サイト）
2. 必要情報一括削除（既存パターンの流用）