# Claude Code向け チェックボックス一括削除機能 追加指示書

## 🎯 機能概要
テストデータが膨大で個別削除が大変なため、チェックボックスで複数選択して一括削除できる機能を追加してください。

## 📋 追加する機能

### 対象ページ
- **お知らせ管理**（/admin - news section）
- **行程表管理**（/admin - schedules section）  
- **参加者管理**（/admin - participants section）

### UI要件
1. **全選択チェックボックス** - ヘッダーに配置、全項目を一括選択/解除
2. **個別チェックボックス** - 各行に配置、個別選択
3. **一括削除ボタン** - 選択されたアイテムをまとめて削除
4. **選択数表示** - 「3件選択中」のような選択状況表示

## 🎨 UI設計例

### テーブルヘッダー部分
```html
<!-- お知らせ管理テーブルの例 -->
<div class="bulk-actions">
    <label>
        <input type="checkbox" id="selectAll" />
        全選択
    </label>
    <button id="bulkDeleteBtn" class="btn btn-danger" disabled>
        選択したアイテムを削除 (<span id="selectedCount">0</span>件)
    </button>
</div>

<table class="table">
    <thead>
        <tr>
            <th width="50px">
                <input type="checkbox" id="selectAllInTable" />
            </th>
            <th>タイトル</th>
            <th>作成日</th>
            <th>操作</th>
        </tr>
    </thead>
    <tbody>
        <!-- データ行 -->
    </tbody>
</table>
```

### テーブル行部分
```html
<tr>
    <td>
        <input type="checkbox" class="item-checkbox" value="<%- item.id %>" />
    </td>
    <td><%- item.title %></td>
    <td><%- item.created_at %></td>
    <td>
        <button class="btn btn-sm btn-primary edit-btn" data-id="<%- item.id %>">
            編集
        </button>
        <button class="btn btn-sm btn-danger delete-btn" data-id="<%- item.id %>">
            削除
        </button>
    </td>
</tr>
```

## 🔧 実装要件

### フロントエンド（JavaScript）

#### 1. チェックボックス制御
```javascript
// 全選択機能
document.getElementById('selectAll').addEventListener('change', function() {
    const checkboxes = document.querySelectorAll('.item-checkbox');
    checkboxes.forEach(checkbox => {
        checkbox.checked = this.checked;
    });
    updateBulkDeleteButton();
});

// 個別チェックボックス制御
document.querySelectorAll('.item-checkbox').forEach(checkbox => {
    checkbox.addEventListener('change', updateBulkDeleteButton);
});

// 選択状況の更新
function updateBulkDeleteButton() {
    const checkedBoxes = document.querySelectorAll('.item-checkbox:checked');
    const count = checkedBoxes.length;
    const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
    const selectedCount = document.getElementById('selectedCount');
    
    selectedCount.textContent = count;
    bulkDeleteBtn.disabled = count === 0;
    
    // 全選択チェックボックスの状態更新
    const allCheckboxes = document.querySelectorAll('.item-checkbox');
    const selectAllCheckbox = document.getElementById('selectAll');
    
    if (count === 0) {
        selectAllCheckbox.indeterminate = false;
        selectAllCheckbox.checked = false;
    } else if (count === allCheckboxes.length) {
        selectAllCheckbox.indeterminate = false;
        selectAllCheckbox.checked = true;
    } else {
        selectAllCheckbox.indeterminate = true;
        selectAllCheckbox.checked = false;
    }
}
```

#### 2. 一括削除処理
```javascript
// 一括削除ボタンのイベント
document.getElementById('bulkDeleteBtn').addEventListener('click', function() {
    const checkedBoxes = document.querySelectorAll('.item-checkbox:checked');
    const ids = Array.from(checkedBoxes).map(checkbox => checkbox.value);
    
    if (ids.length === 0) {
        alert('削除するアイテムを選択してください');
        return;
    }
    
    const confirmMessage = `選択した${ids.length}件のアイテムを削除しますか？\nこの操作は取り消せません。`;
    
    if (confirm(confirmMessage)) {
        bulkDelete(ids);
    }
});

// 一括削除API呼び出し
function bulkDelete(ids) {
    const bulkDeleteBtn = document.getElementById('bulkDeleteBtn');
    const originalText = bulkDeleteBtn.innerHTML;
    
    // ローディング表示
    bulkDeleteBtn.innerHTML = '削除中...';
    bulkDeleteBtn.disabled = true;
    
    fetch('/admin/api/bulk-delete', {
        method: 'DELETE',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            type: 'news', // 'schedules', 'participants'に応じて変更
            ids: ids
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert(`${data.deletedCount}件のアイテムを削除しました`);
            location.reload(); // ページリロード
        } else {
            alert('削除に失敗しました: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('削除処理でエラーが発生しました');
    })
    .finally(() => {
        // ボタンを元に戻す
        bulkDeleteBtn.innerHTML = originalText;
        bulkDeleteBtn.disabled = false;
    });
}
```

### バックエンド（API）

#### 一括削除エンドポイント
```javascript
// routes/admin.js に追加

// 一括削除API
router.delete('/api/bulk-delete', async (req, res) => {
    try {
        const { type, ids } = req.body;
        
        if (!ids || !Array.isArray(ids) || ids.length === 0) {
            return res.status(400).json({ 
                success: false, 
                message: '削除するIDが指定されていません' 
            });
        }
        
        // テーブル名の検証（SQLインジェクション対策）
        const allowedTypes = ['news', 'schedules', 'participants'];
        if (!allowedTypes.includes(type)) {
            return res.status(400).json({ 
                success: false, 
                message: '無効なテーブル名です' 
            });
        }
        
        // IDsを安全な形式に変換
        const safeIds = ids.map(id => parseInt(id)).filter(id => !isNaN(id));
        if (safeIds.length === 0) {
            return res.status(400).json({ 
                success: false, 
                message: '有効なIDがありません' 
            });
        }
        
        // プレースホルダーを作成（?,?,?...）
        const placeholders = safeIds.map(() => '?').join(',');
        const query = `DELETE FROM ${type} WHERE id IN (${placeholders})`;
        
        // データベース実行
        const result = await db.run(query, safeIds);
        
        res.json({ 
            success: true, 
            deletedCount: result.changes,
            message: `${result.changes}件のアイテムを削除しました`
        });
        
    } catch (error) {
        console.error('Bulk delete error:', error);
        res.status(500).json({ 
            success: false, 
            error: error.message 
        });
    }
});

// 参加者の場合は関連データも削除
router.delete('/api/bulk-delete-participants', async (req, res) => {
    try {
        const { ids } = req.body;
        const safeIds = ids.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        // トランザクション処理
        await db.run('BEGIN TRANSACTION');
        
        try {
            // プロフィール削除
            const placeholders = safeIds.map(() => '?').join(',');
            await db.run(`DELETE FROM student_profiles WHERE participant_id IN (${placeholders})`, safeIds);
            
            // 参加者削除
            const result = await db.run(`DELETE FROM participants WHERE id IN (${placeholders})`, safeIds);
            
            await db.run('COMMIT');
            
            res.json({ 
                success: true, 
                deletedCount: result.changes 
            });
            
        } catch (error) {
            await db.run('ROLLBACK');
            throw error;
        }
        
    } catch (error) {
        console.error('Bulk delete participants error:', error);
        res.status(500).json({ success: false, error: error.message });
    }
});
```

## 🎨 CSS スタイリング

### 推奨スタイル
```css
/* 一括操作エリア */
.bulk-actions {
    margin-bottom: 15px;
    padding: 10px;
    background-color: #f8f9fa;
    border-radius: 5px;
    display: flex;
    align-items: center;
    gap: 15px;
}

.bulk-actions label {
    display: flex;
    align-items: center;
    gap: 5px;
    margin: 0;
}

/* チェックボックスのスタイリング */
.item-checkbox, #selectAll {
    transform: scale(1.2);
    margin-right: 5px;
}

/* 一括削除ボタン */
#bulkDeleteBtn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

/* 選択数表示 */
#selectedCount {
    font-weight: bold;
    color: #dc3545;
}

/* テーブルヘッダーのチェックボックス */
th input[type="checkbox"] {
    transform: scale(1.1);
}

/* ホバー効果 */
.table tbody tr:hover {
    background-color: #f8f9fa;
}

.table tbody tr:has(.item-checkbox:checked) {
    background-color: #e3f2fd;
}
```

## ✅ テスト項目

### 基本動作テスト
1. **チェックボックス動作**
   - [ ] 個別チェックボックスの選択/解除
   - [ ] 全選択チェックボックスの動作
   - [ ] 部分選択時のインデターミネイト表示

2. **一括削除テスト**
   - [ ] 複数アイテム選択→削除実行
   - [ ] 削除確認ダイアログの表示
   - [ ] 削除完了後の画面更新

3. **エラーハンドリング**
   - [ ] 未選択で削除ボタンクリック時の動作
   - [ ] API エラー時の適切なメッセージ表示

### UX テスト
- [ ] 選択数のリアルタイム更新
- [ ] ローディング状態の表示
- [ ] 削除完了時のフィードバック

## 🚀 実装優先順位

### Phase 1: お知らせの一括削除
- 最もシンプルな構造で実装・テスト

### Phase 2: 行程表の一括削除
- お知らせと同様の構造で実装

### Phase 3: 参加者の一括削除
- 関連テーブル（student_profiles）も考慮した実装

## 📝 完了報告

実装完了時に以下を報告してください：
1. **実装した機能の詳細**
2. **各テーブル（news/schedules/participants）での動作確認**
3. **UI/UXの動作確認結果**
4. **大量データでのパフォーマンステスト結果**

---

**注意:** この機能は開発・テスト段階での利便性向上が目的です。本番運用時には削除権限の管理を慎重に行ってください。