# Claude Code向け 削除機能修正 指示書

## 🎯 修正概要
編集機能修正後、削除ボタンが実際には動作していないことが判明。削除機能を完全に動作させる修正をお願いします。

## 📋 現在の状況

### 動作確認済み
- ✅ 編集ボタン（fix001.mdで修正完了）
- ✅ サイト基本表示・管理画面ログイン

### 🚨 修正が必要
- ❌ **削除ボタンが実際には動作していない**
  - クリック反応はあるが、データベースから削除されていない
  - ページリロード後も削除対象が残っている

## 🔍 問題の詳細分析

### 考えられる原因
1. **フロントエンドのみの削除** - DOM要素は消えるがDB削除されない
2. **API呼び出し失敗** - DELETE リクエストが送信されていない/失敗
3. **データベース削除処理未実装** - API は動くがSQL削除が実行されない
4. **トランザクション問題** - 削除処理が正常にコミットされない

## 🔧 修正指示

### 1. 削除機能の動作確認方法

#### テスト手順
```bash
# 開発サーバー起動
cd ~/Developer/projects/interact-travel-site
npm run dev

# ブラウザで以下を確認
# 1. http://localhost:3000/admin にログイン
# 2. お知らせ一覧で削除ボタンクリック
# 3. ページリロード後、削除対象がまだ存在するか確認
# 4. ブラウザの開発者ツール(F12)でNetworkタブを確認
```

### 2. 確認・修正すべきポイント

#### A. フロントエンド（JavaScript）
```javascript
// 削除ボタンのイベント処理を確認
// public/js/admin.js 等で以下のような処理があるか

// 正しい削除処理の例
function deleteItem(id, type) {
    if (confirm('本当に削除しますか？')) {
        fetch(`/admin/api/${type}/${id}`, {
            method: 'DELETE',
            headers: {
                'Content-Type': 'application/json',
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // DOM要素削除 + ページリロード
                location.reload();
            } else {
                alert('削除に失敗しました');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('削除処理でエラーが発生しました');
        });
    }
}
```

#### B. バックエンド（API）
```javascript
// routes/admin.js で削除エンドポイントを確認

// お知らせ削除の例
router.delete('/api/news/:id', async (req, res) => {
    try {
        const id = req.params.id;
        
        // データベースから実際に削除
        const result = await db.run('DELETE FROM news WHERE id = ?', [id]);
        
        if (result.changes > 0) {
            res.json({ success: true, message: '削除しました' });
        } else {
            res.json({ success: false, message: '削除対象が見つかりません' });
        }
    } catch (error) {
        console.error('Delete error:', error);
        res.status(500).json({ success: false, error: error.message });
    }
});

// 行程表削除の例
router.delete('/api/schedules/:id', async (req, res) => {
    // 同様の削除処理
});

// 参加者削除の例
router.delete('/api/participants/:id', async (req, res) => {
    // 同様の削除処理
});
```

#### C. データベース接続
```javascript
// models/database.js でDB接続が正常か確認
// トランザクション処理が必要な場合の例

async function deleteWithTransaction(table, id) {
    const db = await getDatabase();
    try {
        await db.run('BEGIN TRANSACTION');
        const result = await db.run(`DELETE FROM ${table} WHERE id = ?`, [id]);
        await db.run('COMMIT');
        return result;
    } catch (error) {
        await db.run('ROLLBACK');
        throw error;
    }
}
```

### 3. デバッグ手順

#### Step 1: ネットワーク確認
```
1. ブラウザでF12を押して開発者ツール開く
2. Networkタブを選択
3. 削除ボタンをクリック
4. DELETE リクエストが送信されているか確認
5. レスポンスコードとレスポンス内容を確認
```

#### Step 2: サーバーログ確認
```bash
# ターミナルでサーバーログを確認
# DELETE リクエストが届いているか
# エラーメッセージが出ていないか
```

#### Step 3: データベース直接確認
```javascript
// SQLite データベースを直接確認
// scripts/check-db.js を作成して確認

const sqlite3 = require('sqlite3').verbose();
const db = new sqlite3.Database('./database.sqlite');

db.all('SELECT * FROM news', (err, rows) => {
    console.log('News:', rows);
});

db.all('SELECT * FROM schedules', (err, rows) => {
    console.log('Schedules:', rows);
});

db.all('SELECT * FROM participants', (err, rows) => {
    console.log('Participants:', rows);
});
```

## ✅ 修正完了後の確認項目

### テストシナリオ
1. **お知らせ削除テスト**
   - [ ] 管理画面でお知らせ削除ボタンクリック
   - [ ] 確認ダイアログが表示される
   - [ ] 「OK」で実際にデータが削除される
   - [ ] ページリロード後、削除したお知らせが表示されない

2. **行程表削除テスト**
   - [ ] 行程表の削除ボタンクリック
   - [ ] 実際にデータが削除される

3. **参加者削除テスト**
   - [ ] 参加者の削除ボタンクリック
   - [ ] 実際にデータが削除される

### エラーハンドリング確認
- [ ] 削除失敗時に適切なエラーメッセージ表示
- [ ] ネットワークエラー時の処理
- [ ] 削除対象が存在しない場合の処理

## 🚨 重要な注意事項

### データ整合性
- 参加者削除時は関連するプロフィール情報も削除
- 外部キー制約がある場合の処理

### セキュリティ
- 削除権限の確認（管理者のみ）
- SQLインジェクション対策

### UX改善
- 削除確認ダイアログ
- 削除成功時のフィードバック
- ローディング表示

## 📝 完了報告項目

修正完了時に以下を報告してください：
1. **修正した具体的な箇所**
2. **削除機能のテスト結果**
3. **各テーブル（news/schedules/participants）での動作確認結果**
4. **エラーハンドリングの動作確認**

---

**優先順位:** まずは基本的な削除機能を確実に動作させることを最優先してください。