# Claude Code 実装プロンプト - fix013対応

## 📌 作業概要

2590地区インターアクト研修旅行サイトの本番環境移行前に、以下2つの緊急修正を実施してください：

1. **セキュリティ修正**: 学生プロフィール編集の認証強化（最優先）
2. **機能追加**: 学生の班をG班まで拡張（A～Gの7班対応）

---

## 🚨 修正1: 学生プロフィール編集の認証強化（最優先）

### 現在の問題
- 学生の詳細画面(`/student/:id`)から編集ボタンを経由すると、**ログインなしで**プロフィール編集ページにアクセスできてしまう
- 未認証ユーザーによる学生プロフィールの不正改ざんリスクがある

### 期待される動作
- 未認証ユーザーには「ログインして編集」ボタンのみ表示
- 編集ページへの直接アクセスは認証必須（未ログインの場合はログインページへリダイレクト）
- 学生本人または管理者のみが編集可能
- 他の学生のプロフィール編集は403エラー

### 実装内容

#### ファイル1: `routes/student.js`

**修正箇所1**: `requireStudentAuth` ミドルウェアの修正
```javascript
// 既存のrequireStudentAuthを以下のコードに完全置き換え
function requireStudentAuth(req, res, next) {
  const studentId = parseInt(req.params.id);
  
  // 管理者の場合は全学生のプロフィール編集可能
  if (req.session.isAdmin) {
    return next();
  }
  
  // 学生本人の場合のみ編集可能
  if (req.session.studentId && req.session.studentId === studentId) {
    return next();
  }
  
  // 未認証の場合はログインページへリダイレクト（編集ページのURLを保持）
  return res.redirect(`/student/login?redirect=/student/${studentId}/edit`);
}
```

**修正箇所2**: プロフィール表示ページ（GET /student/:id）の権限チェック追加
```javascript
// 既存のGET /student/:idルートを以下のコードに置き換え
router.get('/:id', (req, res) => {
  const studentId = parseInt(req.params.id);
  
  db.get(`
    SELECT p.*, s.participation_comment, s.hobby_selfpr, s.prestudy_task 
    FROM participants p
    LEFT JOIN student_profiles s ON p.id = s.student_id
    WHERE p.id = ? AND p.type = 'student'
  `, [studentId], (err, student) => {
    if (err) {
      return res.status(500).render('error', { 
        message: 'データベースエラーが発生しました',
        error: err 
      });
    }
    
    if (!student) {
      return res.status(404).render('error', { 
        message: '学生が見つかりません',
        error: { status: 404 }
      });
    }
    
    // 編集権限の判定（管理者または本人）
    const canEdit = req.session.isAdmin || 
                   (req.session.studentId && req.session.studentId === studentId);
    
    // ログイン状態の判定
    const isLoggedIn = req.session.isAdmin || req.session.studentId;
    
    res.render('student-profile', { 
      student, 
      canEdit,
      isLoggedIn,
      currentStudentId: req.session.studentId || null,
      req: req  // テンプレート内でreq.sessionにアクセスするため
    });
  });
});
```

**確認事項**: 編集ページと更新処理に`requireStudentAuth`が適用されているか確認
```javascript
// 以下の2つのルートにrequireStudentAuthミドルウェアが適用されているか確認

// GET /student/:id/edit
router.get('/:id/edit', requireStudentAuth, (req, res) => {
  // 既存のコード
});

// POST /student/:id/update
router.post('/:id/update', requireStudentAuth, (req, res) => {
  // 既存のコード
});
```

#### ファイル2: `views/student-profile.ejs`

**修正箇所**: 編集ボタンの条件分岐を厳密化

編集ボタン部分を以下のコードに置き換えてください：
```ejs
<!-- アクションボタンエリア -->
<div class="action-buttons">
  <% if (typeof canEdit !== 'undefined' && canEdit) { %>
    <!-- 編集権限がある場合（本人ログイン済みまたは管理者） -->
    <a href="/student/<%= student.id %>/edit" class="btn btn-primary">
      <i class="bi bi-pencil"></i> プロフィールを編集
    </a>
    
    <% if (typeof isLoggedIn !== 'undefined' && isLoggedIn && typeof req !== 'undefined' && !req.session.isAdmin) { %>
      <!-- 学生本人がログインしている場合のみログアウトボタン表示 -->
      <a href="/student/logout" class="btn btn-secondary">
        <i class="bi bi-box-arrow-right"></i> ログアウト
      </a>
    <% } %>
  <% } else { %>
    <!-- 編集権限がない場合 -->
    <a href="/student/login?redirect=/student/<%= student.id %>/edit" class="btn btn-primary">
      <i class="bi bi-box-arrow-in-right"></i> ログインして編集
    </a>
  <% } %>
  
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-arrow-left"></i> トップページに戻る
  </a>
</div>
```

#### ファイル3: `views/student/login.ejs`

**確認事項**: redirect パラメータが正しく処理されているか確認

フォーム内に以下のhidden inputがあることを確認してください：
```ejs
<input type="hidden" name="redirect" value="<%= typeof redirect !== 'undefined' ? redirect : '' %>">
```

もしない場合は、フォームの最後（送信ボタンの直前）に追加してください。

---

## ✅ 修正2: G班の追加

### 現在の問題
- 学生の班がA班～F班の6班のみ
- G班の学生を追加できない

### 実装内容

#### ファイル1: `views/admin/participants.ejs`

**修正箇所**: 班の選択肢にG班を追加

参加者追加フォームと編集モーダルの両方で、以下のように班の選択肢を修正してください：

```html
<!-- 参加者追加フォーム内の班選択 -->
<div class="form-group" id="groupField">
  <label for="group">班 <span style="color: red;">*</span></label>
  <select id="group" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>

<!-- 編集モーダル内の班選択も同様に -->
<div class="form-group" id="editGroupField">
  <label for="editGroup">班 <span style="color: red;">*</span></label>
  <select id="editGroup" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>
```

#### ファイル2: `routes/index.js`

**修正箇所**: G班のデータを取得・表示できるように修正

メインページのルーティング（GET /）内で、班の配列にGを追加してください：

```javascript
router.get('/', (req, res) => {
  // ... 既存のコード（お知らせ、行程表などの取得） ...
  
  // 学生を班別にグループ化（A～Gまで対応）
  const groups = ['A', 'B', 'C', 'D', 'E', 'F', 'G'];  // ← Gを追加
  const groupedStudents = {};
  
  groups.forEach(group => {
    groupedStudents[group] = students.filter(s => s.group === group);
  });
  
  // ... 残りの既存コード（render部分） ...
});
```

#### ファイル3: `views/index.ejs`

**確認事項**: G班が自動的に表示されるか確認

`index.ejs`内の学生表示部分を確認してください。以下のように動的に班を表示している場合は**修正不要**です：

```ejs
<!-- このようなコードがあれば修正不要（動的に班を表示） -->
<% Object.keys(groupedStudents).forEach(group => { %>
  <% if (groupedStudents[group].length > 0) { %>
    <!-- 班ごとの表示 -->
  <% } %>
<% }) %>
```

もし以下のように固定で記述されている場合は、G班を追加してください：

```ejs
<!-- もしこのような固定記述がある場合 -->
<% ['A', 'B', 'C', 'D', 'E', 'F', 'G'].forEach(group => { %>  <!-- ← Gを追加 -->
  <% if (groupedStudents[group] && groupedStudents[group].length > 0) { %>
    <div class="group-section">
      <h3 class="group-title"><%= group %>班</h3>
      <!-- 学生カード表示 -->
    </div>
  <% } %>
<% }) %>
```

---

## 🧪 動作確認テスト

### セキュリティ修正のテスト

修正完了後、以下のテストを必ず実施してください：

#### テスト1: 未ログイン状態での編集ボタン確認
```bash
# ブラウザのシークレットモードで以下にアクセス
http://localhost:3000/student/1
```
- **期待結果**: 「ログインして編集」ボタンが表示される
- **期待結果**: 「プロフィールを編集」ボタンは表示されない

#### テスト2: 未ログイン状態での編集ページ直接アクセス
```bash
# シークレットモードで以下に直接アクセス
http://localhost:3000/student/1/edit
```
- **期待結果**: `/student/login?redirect=/student/1/edit`にリダイレクトされる
- **期待結果**: 編集ページは表示されない

#### テスト3: ログイン後の編集ボタン確認
1. 学生1としてログイン（例: ID=1のパスワードを使用）
2. `http://localhost:3000/student/1`にアクセス
- **期待結果**: 「プロフィールを編集」ボタンが表示される
- **期待結果**: 「ログアウト」ボタンが表示される

#### テスト4: 他の学生のプロフィール編集試行
1. 学生1としてログイン
2. `http://localhost:3000/student/2/edit`にアクセス
- **期待結果**: ログインページにリダイレクトまたは403エラー

#### テスト5: 管理者権限での編集確認
1. 管理者としてログイン（admin / dev2590）
2. `http://localhost:3000/admin`から任意の学生の編集ページにアクセス
- **期待結果**: 全ての学生の編集が可能

### G班追加のテスト

#### テスト6: 管理画面でG班の参加者を追加
1. 管理画面（`http://localhost:3000/admin`）にログイン
2. 参加者管理ページで「参加者を追加」をクリック
3. 班の選択肢に「G班」があることを確認
4. テストでG班の学生を1人追加
- **期待結果**: G班の学生が正常に追加される

#### テスト7: メインページでG班が表示されるか確認
```bash
http://localhost:3000
```
- **期待結果**: 参加者セクションに「G班」が表示される
- **期待結果**: G班の学生が正しく表示される

#### テスト8: G班の学生を編集
1. 管理画面でG班学生の編集をクリック
2. 班の選択肢で「G班」が選択されていることを確認
- **期待結果**: 編集フォームが正しく動作する

---

## 📁 修正対象ファイル一覧

以下のファイルを修正してください：

### セキュリティ修正
- [ ] `routes/student.js`
- [ ] `views/student-profile.ejs`
- [ ] `views/student/login.ejs`（確認のみ）

### G班追加
- [ ] `views/admin/participants.ejs`
- [ ] `routes/index.js`
- [ ] `views/index.ejs`（確認・必要に応じて修正）

---

## ⚠️ 重要な注意事項

1. **必ずバックアップを取ってから作業してください**
   ```bash
   npm run backup-db  # データベースバックアップ
   ```

2. **セキュリティ修正を最優先で実施してください**
   - G班追加よりも先に認証強化を完了させてください
   - 本番環境移行前に必須の修正です

3. **テストは必ずシークレットモードで実施してください**
   - 通常ブラウザだとセッションが残っている可能性があります
   - 正確なテスト結果を得るためシークレットモードを使用してください

4. **サーバーの再起動**
   ```bash
   # 修正後は必ずサーバーを再起動してください
   npm run dev
   ```

5. **データベースへの影響**
   - G班追加はデータベーススキーマの変更は不要です
   - groupフィールドは文字列型なのでA～Gまで対応可能です

---

## 📋 完了チェックリスト

修正完了後、以下をチェックしてください：

### セキュリティ修正
- [ ] `routes/student.js`の`requireStudentAuth`修正完了
- [ ] プロフィール表示ページの権限チェック追加完了
- [ ] `student-profile.ejs`の条件分岐修正完了
- [ ] テスト1～5の全てがパス

### G班追加
- [ ] 管理画面の班選択肢にG班追加完了
- [ ] メインページの班配列にG追加完了
- [ ] `index.ejs`の確認完了
- [ ] テスト6～8の全てがパス

### 総合確認
- [ ] サーバーがエラーなく起動
- [ ] 管理画面が正常動作
- [ ] メインページが正常表示
- [ ] 学生ログイン機能が正常動作
- [ ] 全てのテストケースがパス

---

## 🎯 完了後の報告事項

修正完了後、以下を報告してください：

1. **修正したファイルのリスト**
2. **全テストケースの結果**
3. **発生した問題とその解決方法**（あれば）
4. **追加で必要な作業**（あれば）

---

## 📞 質問・不明点があれば

実装中に不明点や問題が発生した場合は、以下を確認してください：

1. `fix013.md`の詳細な修正指示書
2. `history.md`の過去の修正履歴
3. 既存のコードのコメント

それでも解決しない場合は、エラーメッセージと状況を詳しく報告してください。

---

**本番環境移行前の最終修正です。丁寧にテストを実施してください。**
