# Claude Code 実装プロンプト - fix013対応（修正版）

## 📌 作業概要

2590地区インターアクト研修旅行サイトの本番環境移行前に、以下2つの修正を実施してください：

1. **仕様変更**: 学生プロフィール編集を「メインページ下部のログインフォームからのみ」に変更
2. **機能追加**: 学生の班をG班まで拡張（A～Gの7班対応）

---

## 🔄 修正1: 学生プロフィール編集の仕様変更

### 変更の背景
- 参加者詳細画面からの編集機能でエラーが発生している
- より**シンプルで分かりやすい導線**に変更

### 新しい仕様
- **参加者詳細画面 (`/student/:id`)**: 閲覧専用（編集ボタンなし）
- **メインページ下部**: 学生ログインフォームを設置
- **ログイン後**: 自動的に自分の編集ページ (`/student/:id/edit`) にリダイレクト
- **編集完了後**: ログアウトしてメインページに戻る

### この変更のメリット
✅ シンプルな導線で混乱を防ぐ  
✅ エラーの発生を防止  
✅ セキュリティリスクを低減  
✅ 学生にとってわかりやすい操作方法

---

## 📝 実装内容

### ファイル1: `routes/student.js`

#### 修正箇所1: プロフィール表示ページをシンプルに変更

**既存のGET /student/:idルートを以下に置き換えてください：**

```javascript
// GET /student/:id - プロフィール表示ページ（閲覧専用）
router.get('/:id', (req, res) => {
  const studentId = parseInt(req.params.id);
  
  db.get(`
    SELECT p.*, s.participation_comment, s.hobby_selfpr, s.prestudy_task 
    FROM participants p
    LEFT JOIN student_profiles s ON p.id = s.student_id
    WHERE p.id = ? AND p.type = 'student'
  `, [studentId], (err, student) => {
    if (err) {
      return res.status(500).render('error', { 
        message: 'データベースエラーが発生しました',
        error: err 
      });
    }
    
    if (!student) {
      return res.status(404).render('error', { 
        message: '学生が見つかりません',
        error: { status: 404 }
      });
    }
    
    // 閲覧専用ページなので、編集関連の変数は渡さない
    res.render('student-profile', { 
      student
    });
  });
});
```

**ポイント**: `canEdit`、`isLoggedIn`などの編集関連変数を削除し、シンプルに`student`データのみを渡す。

#### 修正箇所2: ログイン処理を修正（ログイン後に編集ページへリダイレクト）

**既存のPOST /student/loginルートを以下のように修正してください：**

```javascript
// POST /student/login - ログイン処理
router.post('/login', (req, res) => {
  const { studentId, password } = req.body;
  
  if (!studentId || !password) {
    return res.render('student/login', { 
      error: '学生IDとパスワードを入力してください',
      redirect: '' 
    });
  }
  
  db.get(
    'SELECT * FROM participants WHERE id = ? AND type = "student"',
    [studentId],
    async (err, student) => {
      if (err) {
        return res.render('student/login', { 
          error: 'ログインに失敗しました',
          redirect: '' 
        });
      }
      
      if (!student) {
        return res.render('student/login', { 
          error: '学生IDまたはパスワードが間違っています',
          redirect: '' 
        });
      }
      
      try {
        const match = await bcrypt.compare(password, student.password);
        if (match) {
          req.session.studentId = student.id;
          // ログイン成功後、自分の編集ページへリダイレクト
          return res.redirect(`/student/${student.id}/edit`);
        } else {
          return res.render('student/login', { 
            error: '学生IDまたはパスワードが間違っています',
            redirect: '' 
          });
        }
      } catch (error) {
        return res.render('student/login', { 
          error: 'ログインに失敗しました',
          redirect: '' 
        });
      }
    }
  );
});
```

**変更点**: ログイン成功後のリダイレクト先を`/student/${student.id}/edit`に変更。

#### 修正箇所3: `requireStudentAuth` ミドルウェアの修正

**既存のrequireStudentAuth関数を以下のように修正してください：**

```javascript
function requireStudentAuth(req, res, next) {
  const studentId = parseInt(req.params.id);
  
  // 管理者の場合は全学生のプロフィール編集可能
  if (req.session.isAdmin) {
    return next();
  }
  
  // 学生本人の場合のみ編集可能
  if (req.session.studentId && req.session.studentId === studentId) {
    return next();
  }
  
  // 未認証の場合はメインページへリダイレクト
  return res.redirect('/?error=unauthorized');
}
```

**変更点**: 未認証時のリダイレクト先をメインページに変更。

#### 確認事項: 編集ページと更新処理に認証が適用されているか

以下のルートに`requireStudentAuth`ミドルウェアが適用されていることを確認してください：

```javascript
// GET /student/:id/edit - 編集ページ（認証必須）
router.get('/:id/edit', requireStudentAuth, (req, res) => {
  // 既存のコード（変更不要）
});

// POST /student/:id/update - 更新処理（認証必須）
router.post('/:id/update', requireStudentAuth, (req, res) => {
  // 既存のコード（変更不要）
});
```

---

### ファイル2: `views/student-profile.ejs`

#### 修正: 編集ボタンを完全に削除

**アクションボタンエリアを以下のコードに置き換えてください：**

```ejs
<!-- アクションボタンエリア - 編集機能を削除し、閲覧専用に -->
<div class="action-buttons" style="text-align: center; margin-top: 30px;">
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-arrow-left"></i> トップページに戻る
  </a>
</div>

<!-- 学生ログインへの誘導メッセージ -->
<div style="text-align: center; margin-top: 20px; padding: 15px; background: #f8f9fa; border-radius: 8px;">
  <p style="margin: 0; color: #666; font-size: 14px;">
    <i class="bi bi-info-circle"></i> 
    プロフィールを編集する場合は、
    <a href="/#student-login" style="color: #2590ff; font-weight: bold;">メインページ下部の学生ログイン</a>
    からログインしてください。
  </p>
</div>
```

**ポイント**: 
- 編集ボタンをすべて削除
- 「トップページに戻る」ボタンのみ残す
- ログインへの誘導メッセージを追加

---

### ファイル3: `views/student/edit.ejs`

#### 修正: ログアウトボタンとナビゲーションボタンの追加

**編集ページの保存ボタンの下（フォームの外）に以下を追加してください：**

```ejs
<!-- 編集ページのアクションボタン -->
<div style="margin-top: 30px; text-align: center;">
  <button type="submit" form="editForm" class="btn btn-primary">
    <i class="bi bi-check-circle"></i> 保存
  </button>
  <a href="/student/logout" class="btn btn-secondary">
    <i class="bi bi-box-arrow-right"></i> ログアウト
  </a>
  <a href="/" class="btn btn-secondary">
    <i class="bi bi-house"></i> トップページに戻る
  </a>
</div>
```

**ポイント**: 
- 保存ボタン
- ログアウトボタン
- トップページに戻るボタンの3つを配置

---

### ファイル4: `views/index.ejs`

#### 確認: 学生ログインフォームが存在することを確認

メインページ下部（フッターの前）に以下のような学生ログインセクションがあることを確認してください：

```ejs
<!-- 学生ログインセクション -->
<section id="student-login" class="section">
  <div class="container">
    <h2><i class="bi bi-person-circle"></i> 学生ログイン</h2>
    <p>プロフィール編集を行う学生の方は、以下からログインしてください。</p>
    
    <% if (typeof error !== 'undefined' && error) { %>
      <div class="alert alert-danger"><%= error %></div>
    <% } %>
    
    <form action="/student/login" method="POST" style="max-width: 400px; margin: 0 auto;">
      <div class="form-group">
        <label for="studentId">学生ID</label>
        <input type="text" id="studentId" name="studentId" class="form-control" required 
               placeholder="例: 1">
      </div>
      <div class="form-group">
        <label for="password">パスワード</label>
        <input type="password" id="password" name="password" class="form-control" required>
      </div>
      <button type="submit" class="btn btn-primary" style="width: 100%;">
        <i class="bi bi-box-arrow-in-right"></i> ログイン
      </button>
    </form>
  </div>
</section>
```

**もしこのセクションがない場合は追加してください。**

---

## ✅ 修正2: G班の追加

### ファイル1: `views/admin/participants.ejs`

#### 修正: 班の選択肢にG班を追加

**参加者追加フォームと編集モーダルの両方で、班の選択肢を修正してください：**

```html
<!-- 参加者追加フォーム内の班選択 -->
<div class="form-group" id="groupField">
  <label for="group">班 <span style="color: red;">*</span></label>
  <select id="group" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>

<!-- 編集モーダル内の班選択も同様に -->
<div class="form-group" id="editGroupField">
  <label for="editGroup">班 <span style="color: red;">*</span></label>
  <select id="editGroup" name="group" required>
    <option value="">選択してください</option>
    <option value="A">A班</option>
    <option value="B">B班</option>
    <option value="C">C班</option>
    <option value="D">D班</option>
    <option value="E">E班</option>
    <option value="F">F班</option>
    <option value="G">G班</option>  <!-- ← 追加 -->
  </select>
</div>
```

---

### ファイル2: `routes/index.js`

#### 修正: G班のデータを取得・表示

**メインページのルーティング（GET /）内で、班の配列にGを追加してください：**

```javascript
router.get('/', (req, res) => {
  // ... 既存のコード（お知らせ、行程表などの取得） ...
  
  // 学生を班別にグループ化（A～Gまで対応）
  const groups = ['A', 'B', 'C', 'D', 'E', 'F', 'G'];  // ← Gを追加
  const groupedStudents = {};
  
  groups.forEach(group => {
    groupedStudents[group] = students.filter(s => s.group === group);
  });
  
  // ... 残りの既存コード（render部分） ...
});
```

**探し方のヒント**: 
- `router.get('/', ...)` を探してください
- その中で `['A', 'B', 'C', 'D', 'E', 'F']` という配列を探してください
- 配列の最後に `'G'` を追加してください

---

### ファイル3: `views/index.ejs`（確認のみ）

#### 確認: G班が自動的に表示されるか確認

参加者表示部分で、班が動的にループ表示されている場合は**修正不要**です。

もし固定で記述されている場合は、以下のように修正してください：

```ejs
<!-- 学生セクション内 -->
<% ['A', 'B', 'C', 'D', 'E', 'F', 'G'].forEach(group => { %>  <!-- ← Gを追加 -->
  <% if (groupedStudents[group] && groupedStudents[group].length > 0) { %>
    <div class="group-section">
      <h3 class="group-title"><%= group %>班</h3>
      <div class="participants-grid">
        <% groupedStudents[group].forEach(student => { %>
          <!-- 学生カード表示 -->
        <% }) %>
      </div>
    </div>
  <% } %>
<% }) %>
```

---

## 🧪 動作確認テスト

修正完了後、以下のテストを必ず実施してください：

### 【仕様変更のテスト】

#### ✅ テスト1: 参加者詳細画面が閲覧専用になっているか
```bash
http://localhost:3000/student/1
```
- **期待**: 編集ボタンが一切表示されない
- **期待**: 「トップページに戻る」ボタンのみ表示
- **期待**: 「ログインして編集してください」というメッセージが表示される

#### ✅ テスト2: メインページから学生ログイン
```bash
http://localhost:3000/#student-login
```
1. 学生ID（例: 1）とパスワードを入力
2. 「ログイン」をクリック
- **期待**: ログイン成功後、自動的に `/student/1/edit` にリダイレクト
- **期待**: 編集ページが表示される

#### ✅ テスト3: 編集ページで編集可能
1. 編集ページで各項目を変更
2. 「保存」ボタンをクリック
- **期待**: 変更が正常に保存される
- **期待**: 「ログアウト」「トップページに戻る」ボタンが表示される

#### ✅ テスト4: 未ログインで編集ページに直接アクセス
```bash
# シークレットモードで直接アクセス
http://localhost:3000/student/1/edit
```
- **期待**: メインページにリダイレクトされる
- **期待**: 編集ページは表示されない

#### ✅ テスト5: 他の学生の編集ページにアクセス
1. 学生1としてログイン
2. 学生2の編集ページにアクセス試行
```bash
http://localhost:3000/student/2/edit
```
- **期待**: アクセスが拒否される（メインページにリダイレクト）

#### ✅ テスト6: ログアウト機能
1. 編集ページの「ログアウト」ボタンをクリック
- **期待**: メインページにリダイレクトされる
- **期待**: 学生ログインフォームが表示される

### 【G班追加のテスト】

#### ✅ テスト7: 管理画面でG班を追加
```bash
http://localhost:3000/admin
```
1. 参加者管理で「参加者を追加」
2. 班の選択肢に「G班」があることを確認
3. G班の学生を追加
- **期待**: G班が選択できる
- **期待**: G班の学生が正常に追加される

#### ✅ テスト8: メインページでG班が表示
```bash
http://localhost:3000
```
- **期待**: 参加者セクションに「G班」が表示される
- **期待**: G班の学生が正しく表示される

---

## 📁 修正対象ファイル一覧

### 仕様変更（優先）
- [ ] `routes/student.js` - 3箇所修正
- [ ] `views/student-profile.ejs` - 編集ボタン削除
- [ ] `views/student/edit.ejs` - ログアウトボタン追加
- [ ] `views/index.ejs` - 学生ログインフォーム確認

### G班追加
- [ ] `views/admin/participants.ejs` - 班選択肢追加
- [ ] `routes/index.js` - 班配列にG追加
- [ ] `views/index.ejs` - G班表示確認

---

## ⚠️ 重要な注意事項

1. **必ずバックアップを取ってください**
   ```bash
   npm run backup-db
   ```

2. **仕様変更を先に完了させてください**
   - G班追加よりも先に学生ログイン周りの修正を完了させてください

3. **サーバーの再起動**
   ```bash
   # 修正後は必ずサーバーを再起動
   npm run dev
   ```

4. **シークレットモードでテスト**
   - 認証関連のテストは必ずシークレットモードで実施してください

5. **管理者権限の確認**
   - 管理者（admin / dev2590）は引き続き全学生の編集が可能です

---

## 📋 完了チェックリスト

### 仕様変更
- [ ] `routes/student.js` 修正完了（3箇所）
- [ ] `student-profile.ejs` 修正完了
- [ ] `student/edit.ejs` 修正完了
- [ ] テスト1～6の全てがパス

### G班追加
- [ ] `admin/participants.ejs` 修正完了
- [ ] `routes/index.js` 修正完了
- [ ] `index.ejs` 確認完了
- [ ] テスト7～8の全てがパス

### 総合確認
- [ ] サーバー正常起動
- [ ] 全機能正常動作
- [ ] 全テストパス
- [ ] エラーなし

---

## 🎯 完了後の報告

修正完了後、以下を報告してください：

1. **修正したファイル名のリスト**
2. **全テストケースの結果** （パス/失敗）
3. **発生した問題とその解決方法** （あれば）

---

**この修正により、より分かりやすく、エラーの少ないシステムになります。頑張ってください！**
