const express = require('express');
const path = require('path');
const session = require('express-session');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const { initializeDatabase } = require('./scripts/database');

const app = express();
const PORT = process.env.PORT || 3000;

// セキュリティミドルウェア
app.use(helmet({
    contentSecurityPolicy: {
        directives: {
            defaultSrc: ["'self'"],
            styleSrc: ["'self'", "'unsafe-inline'", "https://fonts.googleapis.com"],
            fontSrc: ["'self'", "https://fonts.gstatic.com"],
            imgSrc: ["'self'", "data:", "blob:"],
            scriptSrc: ["'self'", "'unsafe-inline'"]
        }
    }
}));

// レート制限
const limiter = rateLimit({
    windowMs: 15 * 60 * 1000, // 15分
    max: 100, // 最大100リクエスト
    message: 'アクセスが集中しています。しばらくしてからお試しください。'
});

// 学生ルート専用のより緩いレート制限
const studentLimiter = rateLimit({
    windowMs: 15 * 60 * 1000, // 15分
    max: 500, // 最大500リクエスト（より緩い制限）
    message: 'アクセスが集中しています。しばらくしてからお試しください。'
});

app.use(limiter);

// ビューエンジン設定
app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

// ミドルウェア
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static(path.join(__dirname, 'public')));

// セッション設定
app.use(session({
    secret: 'interact-travel-site-secret-key-2590',
    resave: false,
    saveUninitialized: false,
    cookie: {
        secure: process.env.NODE_ENV === 'production',
        httpOnly: true,
        maxAge: 24 * 60 * 60 * 1000 // 24時間
    }
}));

// ルート設定
const indexRouter = require('./routes/index');
const adminRouter = require('./routes/admin');
const studentRouter = require('./routes/student');

app.use('/', indexRouter);
app.use('/admin', adminRouter);
app.use('/student', studentLimiter, studentRouter);

// 404エラーハンドリング
app.use((req, res) => {
    res.status(404).render('error', {
        title: 'ページが見つかりません',
        message: 'お探しのページは存在しません。',
        error: { status: 404 }
    });
});

// エラーハンドリング
app.use((err, req, res, next) => {
    console.error(err.stack);
    res.status(err.status || 500).render('error', {
        title: 'エラーが発生しました',
        message: 'システムエラーが発生しました。管理者にお問い合わせください。',
        error: process.env.NODE_ENV === 'production' ? {} : err
    });
});

// サーバー起動
async function startServer() {
    try {
        // データベース初期化
        await initializeDatabase();
        console.log('データベースの初期化が完了しました。');
        
        // サーバー起動
        app.listen(PORT, () => {
            console.log(`\n🚀 サーバーが起動しました！`);
            console.log(`📱 ローカルURL: http://localhost:${PORT}`);
            console.log(`🔧 管理画面: http://localhost:${PORT}/admin`);
            console.log(`👤 管理者ログイン: admin / dev2590\n`);
        });
    } catch (error) {
        console.error('サーバー起動エラー:', error);
        process.exit(1);
    }
}

startServer();